package com.fanjiao.captcha.server;

import cloud.tianai.captcha.common.constant.CaptchaTypeConstant;
import cloud.tianai.captcha.generator.common.constant.SliderCaptchaConstant;
import cloud.tianai.captcha.generator.impl.StandardSliderImageCaptchaGenerator;
import cloud.tianai.captcha.resource.common.model.dto.Resource;
import cloud.tianai.captcha.resource.common.model.dto.ResourceMap;
import cloud.tianai.captcha.resource.impl.DefaultResourceStore;
import cloud.tianai.captcha.resource.impl.provider.ClassPathResourceProvider;
import cloud.tianai.captcha.resource.impl.provider.FileResourceProvider;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import java.io.File;

import static cloud.tianai.captcha.generator.impl.StandardSliderImageCaptchaGenerator.DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH;

@Slf4j
@Component
@EnableConfigurationProperties({ResourceProperties.class})
public class ResourceStore extends DefaultResourceStore {

    private final ResourceProperties prop;

    public ResourceStore(ResourceProperties prop) {
        super();
        this.prop = prop;

        // 滑块验证码 模板 (系统内置)
        ResourceMap template1 = new ResourceMap("default", 4);
        template1.put(SliderCaptchaConstant.TEMPLATE_ACTIVE_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/1/active.png")));
        template1.put(SliderCaptchaConstant.TEMPLATE_FIXED_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/1/fixed.png")));
        ResourceMap template2 = new ResourceMap("default", 4);
        template2.put(SliderCaptchaConstant.TEMPLATE_ACTIVE_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/2/active.png")));
        template2.put(SliderCaptchaConstant.TEMPLATE_FIXED_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/2/fixed.png")));
        // 旋转验证码 模板 (系统内置)
        ResourceMap template3 = new ResourceMap("default", 4);
        template3.put(SliderCaptchaConstant.TEMPLATE_ACTIVE_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, StandardSliderImageCaptchaGenerator.DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/3/active.png")));
        template3.put(SliderCaptchaConstant.TEMPLATE_FIXED_IMAGE_NAME, new Resource(ClassPathResourceProvider.NAME, StandardSliderImageCaptchaGenerator.DEFAULT_SLIDER_IMAGE_TEMPLATE_PATH.concat("/3/fixed.png")));

        //添加模板
        addTemplate(CaptchaTypeConstant.SLIDER, template1);
        addTemplate(CaptchaTypeConstant.SLIDER, template2);
        addTemplate(CaptchaTypeConstant.ROTATE, template3);

        //添加本地资源
        load();

        log.info("本地资源存储器初始化完成 资源总数 {} 个", getAllResourceCount());
    }

    /**
     * 加载资源
     */
    private void load() {
        String root = getResourceRoot();
        log.info("正在初始化本地资源存储器 {}", root);

        //加载公共资源
        addResourcesFromDir(
                root + File.separator + "common",
                CaptchaTypeConstant.CONCAT,
                CaptchaTypeConstant.ROTATE,
                CaptchaTypeConstant.SLIDER
        );

        //加载各个类型特有的资源
        addResourcesFromDir(
                root + File.separator + CaptchaTypeConstant.CONCAT.toLowerCase(),
                CaptchaTypeConstant.CONCAT
        );
        addResourcesFromDir(
                root + File.separator + CaptchaTypeConstant.ROTATE.toLowerCase(),
                CaptchaTypeConstant.ROTATE
        );
        addResourcesFromDir(
                root + File.separator + CaptchaTypeConstant.SLIDER.toLowerCase(),
                CaptchaTypeConstant.SLIDER
        );
    }

    private void addResourcesFromDir(String dirPath, String... types) {
        if (types.length == 0) {
            return;
        }
        if (!isValidDir(dirPath)) return;
        File dir = new File(dirPath);
        File[] files = dir.listFiles(file -> {
            if (!file.isFile()) return false;
            return file.getName().toLowerCase().matches("^.*\\.(?:jpg|jpeg|png)$");
        });
        if (files == null) return;
        for (File file : files) {
            for (String type : types) {
                addResource(type, new Resource(FileResourceProvider.NAME, file.getAbsolutePath(), "default"));
            }
        }
    }

    private boolean isValidDir(String dirPath) {
        if (dirPath == null || dirPath.isEmpty()) return false;
        File dir = new File(dirPath);
        return dir.exists() && dir.isDirectory() && dir.canRead();
    }

    /**
     * 获取资源文件根目录路径
     */
    private String getResourceRoot() {
        String root = prop.getResourceRoot();
        if (root != null && !root.isBlank()) {
            return root;
        }
        return System.getProperty("user.dir") + File.separator + "resources";
    }
}
