package com.fanjiao.captcha.server;

import cloud.tianai.captcha.common.response.ApiResponse;
import cloud.tianai.captcha.common.response.ApiResponseStatusConstant;
import cloud.tianai.captcha.generator.ImageCaptchaGenerator;
import cloud.tianai.captcha.generator.common.model.dto.GenerateParam;
import cloud.tianai.captcha.spring.application.DefaultImageCaptchaApplication;
import cloud.tianai.captcha.spring.autoconfiguration.ImageCaptchaProperties;
import cloud.tianai.captcha.spring.exception.CaptchaValidException;
import cloud.tianai.captcha.spring.store.CacheStore;
import cloud.tianai.captcha.spring.vo.CaptchaResponse;
import cloud.tianai.captcha.spring.vo.ImageCaptchaVO;
import cloud.tianai.captcha.validator.ImageCaptchaValidator;
import cloud.tianai.captcha.validator.common.model.dto.ImageCaptchaTrack;
import com.fanjiao.captcha.server.dto.ExtraGenerateParam;
import com.fanjiao.captcha.server.dto.UseCaptchaData;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Slf4j
@Component
@EnableConfigurationProperties({ImageCaptchaProperties.class})
public class CaptchaApplication extends DefaultImageCaptchaApplication {

    private final ImageCaptchaProperties prop;

    public CaptchaApplication(
            ImageCaptchaGenerator template,
            ImageCaptchaValidator imageCaptchaValidator,
            CacheStore cacheStore,
            ImageCaptchaProperties prop
    ) {
        super(template, imageCaptchaValidator, cacheStore, prop);
        this.prop = prop;
        log.info("自定义验证码应用初始化完成");
    }

    @Override
    public CaptchaResponse<ImageCaptchaVO> generateCaptcha(GenerateParam param) {
        //扩展一下这个生成验证码方法，如果传入的参数是带有校验数据的，那么在生成验证码成功后，把校验数据也写入缓存
        var result = super.generateCaptcha(param);
        if (param instanceof ExtraGenerateParam extraParam) {
            Long expire = prop.getExpire().getOrDefault(param.getType(), prop.getExpire().getOrDefault("default", 20000L));
            String cacheKey = getKey(result.getId()).concat(":verification");
            if (!getCacheStore().setCache(cacheKey, extraParam.getVerification(), expire, TimeUnit.MILLISECONDS)) {
                log.error("缓存验证码校验数据失败， id={}, validData={}", result.getId(), extraParam.getVerification());
                throw new CaptchaValidException(param.getType(), "缓存验证码校验数据失败");
            }
        }
        return result;
    }

    public ApiResponse<?> matching(String id, ImageCaptchaTrack imageCaptchaTrack, @NonNull String ip) {
        var result = super.matching(id, imageCaptchaTrack);

        //不管校验是否成功，都读取并删除额外的校验数据
        var verification = getCacheStore().getAndRemoveCache(getKey(id).concat(":verification"));

        //如果验证结果是失败的就不做后续的校验了
        if (!result.isSuccess()) {
            return result;
        }

        //没有校验数据视为失败
        if (verification == null) {
            return ApiResponse.ofMessage(ApiResponseStatusConstant.BASIC_CHECK_FAIL);
        }

        //IP不匹配视为失败
        if (!ip.equals(verification.get("ip"))) {
            return ApiResponse.ofMessage(ApiResponseStatusConstant.BASIC_CHECK_FAIL);
        }

        //到这就是校验成功了，那么把校验数据写入到二次验证缓存中
        if (!saveSecondaryData(id, verification)) {
            log.error("缓存验证码二次校验数据失败， id={}, validData={}", id, verification);
            throw new CaptchaValidException("", "缓存验证码二次校验数据失败");
        }

        return result;
    }

    /**
     * 写入二次验证校验数据
     *
     * @param key  验证码id
     * @param data 校验数据
     * @return 是否写入成功
     */
    private boolean saveSecondaryData(String key, Map<String, Object> data) {
        var secondary = prop.getSecondary();
        return getCacheStore().setCache(
                secondary.getKeyPrefix().concat(":").concat(key),
                data,
                secondary.getExpire(),
                TimeUnit.MILLISECONDS
        );
    }

    /**
     * 获取二次校验数据
     *
     * @param key 验证码key
     * @return 二次校验数据
     */
    public Map<String, Object> getSecondaryData(String key) {
        if (key == null || key.isEmpty()) return null;
        var secondary = prop.getSecondary();
        return getCacheStore().getAndRemoveCache(secondary.getKeyPrefix().concat(":").concat(key));
    }

    /**
     * 二次校验（使用）验证码
     *
     * @param data 验证数据
     * @return 是否校验成功
     */
    public boolean secondaryVerification(UseCaptchaData data) {
        if (data == null) return false;

        //读取二次校验数据
        var verification = getSecondaryData(data.getKey());

        //如果传入了ip，那就校验ip
        String ip = data.getIp();
        if (ip != null && !ip.isEmpty()) {
            if (!ip.equals(verification.get("ip"))) {
                return false;
            }
        }

        Map<String, Object> extra = data.getExtra();
        if (extra == null) {
            extra = new HashMap<>();
        }

        Object orgExtra = verification.get("extra");
        if (orgExtra instanceof Map<?, ?> orgMap) {
            //确定原始校验数据中的每个数据都与新传入的数据匹配
            for (Map.Entry<?, ?> entry : orgMap.entrySet()) {
                Object key = entry.getKey();
                if (!(key instanceof String)) {
                    continue;
                }

                if (!Objects.equals(entry.getValue(), extra.get(key))) {
                    return false;
                }
            }
        }

        return true;
    }
}
