import en from './en.json'
import zhCn from './zh-cn.json'
import zhTw from './zh-tw.json'

/**
 * 可用的翻译数据
 */
const translations: Record<string, Record<string, string>> = {
    en,
    'zh-cn': zhCn,
    'zh-tw': zhTw,
}

/**
 * 可用的语言标识
 */
const availableLanguages = Object.keys(translations)

/**
 * 默认的语言标识
 */
const DEFAULT_LANGUAGE = 'zh-cn'

/**
 * 标准化语言标识
 * @param language
 */
function normalizeLanguage(language: string): string {
    const parts = language.split(/_|-/)
    if (parts.length > 2) {
        parts.splice(1, parts.length - 2)
    }
    return parts.map((t) => t.toLowerCase()).join('-')
}

/**
 * 根据输入值，返回可用的语言
 * @param language
 */
export function getLanguage(language?: string): string {
    if (typeof language === 'string') {
        language = normalizeLanguage(language)
        if (availableLanguages.includes(language)) {
            return language
        }
    }

    //尝试通过navigator.languages获取
    if (navigator && navigator.languages) {
        for (let language of navigator.languages) {
            language = normalizeLanguage(language)
            if (availableLanguages.includes(language)) {
                return language
            }
        }
    }

    return DEFAULT_LANGUAGE
}

/**
 * 获取翻译字典
 * @param language
 */
export function getTranslations(language?: string): Record<string, string> {
    return translations[getLanguage(language)] ?? {}
}

/**
 * 获取翻译函数
 * @param language
 */
export function getTranslator(language?: string): (id: string) => string {
    const translation = getTranslations(language)
    return (id: string) => translation[id] ?? id
}
