package com.fanjiao.captcha.server;

import cloud.tianai.captcha.common.constant.CaptchaTypeConstant;
import com.fanjiao.captcha.server.dto.*;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;

@Slf4j
@RestController
@RequestMapping("/api/captcha")
public class CaptchaController {

    private final static List<String> usableCaptchaTypes;

    static {
        usableCaptchaTypes = Arrays.asList(
                CaptchaTypeConstant.SLIDER,
                CaptchaTypeConstant.ROTATE,
                CaptchaTypeConstant.CONCAT
        );
    }

    @Autowired
    private CaptchaApplication imageCaptchaApplication;

    @RequestMapping("/generate")
    @ResponseBody
    public Response<?> generateCaptcha(
            HttpServletRequest request,
            @RequestBody(required = false) GenerateCaptchaData data
    ) {
        var type = getRandomType();
        var ip = request.getRemoteAddr();
        Map<String, Object> extra = null;
        if (data != null) {
            extra = data.getExtra();
        }
        if (extra == null) {
            extra = new HashMap<>();
        }

        //带校验数据的生成参数
        ExtraGenerateParam extraGenerateParam = new ExtraGenerateParam();
        extraGenerateParam.setType(type);

        //填充额外的校验数据
        var validation = extraGenerateParam.getVerification();
        validation.put("ip", ip);
        validation.put("extra", extra);

        //生成验证码
        var result = imageCaptchaApplication.generateCaptcha(extraGenerateParam);

        return Response.success(GenerateCaptchaResult.from(result));
    }

    /**
     * 获取随机的验证码类型
     *
     * @return 验证码类型
     */
    private String getRandomType() {
        int index = ThreadLocalRandom.current().nextInt(0, usableCaptchaTypes.size());
        return usableCaptchaTypes.get(index);
    }

    /**
     * 校验验证码
     *
     * @param request 请求对象
     * @param data    请求体
     */
    @PostMapping("/check")
    @ResponseBody
    public Response<?> checkCaptcha(
            HttpServletRequest request,
            @RequestBody CheckCaptchaData data
    ) {
        String ip = request.getRemoteAddr();
        //校验验证码
        var result = imageCaptchaApplication.matching(
                data.getKey(),
                data.getData(),
                ip
        );

        //检查实际的验证码校验结果
        if (!result.isSuccess()) {
            //校验失败
            return Response.from(result);
        }

        //校验成功
        return Response.success();
    }

    /**
     * 二次校验（使用）验证码
     *
     * @param data 校验数据
     */
    @PostMapping("/use")
    @ResponseBody
    public Response<?> useCaptcha(
            @RequestBody UseCaptchaData data
    ) {
        if (data == null) {
            return Response.fail("校验失败");
        }

        if (!imageCaptchaApplication.secondaryVerification(data)) {
            return Response.fail("校验失败");
        }

        return Response.success();
    }

    @RequestMapping("/verifyInfo")
    public Response<?> verifyCaptcha(
            @RequestParam(name = "key") String key
    ) {
        var data = imageCaptchaApplication.getSecondaryData(key);
        if (data == null) {
            return Response.fail("未验证的数据");
        } else {
            return Response.success(data);
        }
    }
}
