import { CaptchaDisplayData } from './types'

/**
 * 位置数据
 */
interface Position {
    x: number
    y: number
}

/**
 * 创建验证码实例的配置项
 */
interface CreateCaptchaOptions {
    /**
     * 自动绑定开始拖动事件的元素
     */
    triggers?: HTMLElement[]
    /**
     * 验证码显示数据
     */
    captcha: CaptchaDisplayData
    /**
     * 允许拖动的最大距离
     */
    max: number
    /**
     * 当进行拖动时触发的回调
     * @param x 拖动的距离
     */
    onDrag: (x: number) => void
    /**
     * 当拖动完成需要校验数据时触发的回调
     * @param data 待校验的数据
     * @returns
     */
    onVerify: (data: object) => void
}

/**
 * 验证码实例
 */
export interface CaptchaInstance {
    /**
     * 当开始拖动时调用的方法
     * @param event
     * @returns
     */
    onDragStart: (event: MouseEvent | TouchEvent) => void
    /**
     * 销毁验证码实例，解除所有的事件绑定
     */
    destroy(): void
}

/**
 * 拖动事件追踪数据
 */
interface TrackData extends Position {
    type: 'up' | 'move' | 'down'
    t: number
}

/**
 * 拖动中的数据
 */
interface TouchingData extends Position {
    startTime: number
}

/**
 * 从原生事件中获取坐标
 * @param event
 * @returns
 */
function getCurrentCoordinate(event: any): Position {
    if (event.pageX !== null && event.pageX !== undefined) {
        return {
            x: Math.round(event.pageX),
            y: Math.round(event.pageY),
        }
    }
    let targetTouches
    if (event.changedTouches) {
        // 抬起事件
        targetTouches = event.changedTouches
    } else if (event.targetTouches) {
        // pc 按下事件
        targetTouches = event.targetTouches
    } else if (event.originalEvent && event.originalEvent.targetTouches) {
        // 鼠标触摸事件
        targetTouches = event.originalEvent.targetTouches
    }
    if (
        targetTouches[0].pageX !== null &&
        targetTouches[0].pageX !== undefined
    ) {
        return {
            x: Math.round(targetTouches[0].pageX),
            y: Math.round(targetTouches[0].pageY),
        }
    }
    return {
        x: Math.round(targetTouches[0].clientX),
        y: Math.round(targetTouches[0].clientY),
    }
}

/**
 * 创建验证码实例
 * @param options
 */
function createCaptcha(options: CreateCaptchaOptions): CaptchaInstance {
    const { triggers = [], max, onDrag, onVerify } = options

    let touching: TouchingData | null = null
    let trackList: TrackData[] = []

    const onDragStart = (event: MouseEvent | TouchEvent) => {
        const coords = getCurrentCoordinate(event)
        touching = {
            ...coords,
            startTime: Date.now(),
        }
        trackList.push({
            x: 0,
            y: 0,
            type: 'down',
            t: 0,
        })

        //绑定全局事件
        window.addEventListener('touchmove', onDragMove, { passive: true })
        window.addEventListener('mousemove', onDragMove, { passive: true })
        window.addEventListener('touchend', onDragEnd, { passive: true })
        window.addEventListener('mouseup', onDragEnd, { passive: true })
    }

    const onDragMove = (event: MouseEvent | TouchEvent) => {
        if (!touching) return
        const coords = getCurrentCoordinate(event)
        trackList.push({
            x: coords.x - touching.x,
            y: coords.y - touching.y,
            type: 'move',
            t: Date.now() - touching.startTime,
        })

        const x = Math.max(0, Math.min(max, coords.x - touching.x))
        onDrag(x)
    }

    const onDragEnd = (event: MouseEvent | TouchEvent) => {
        if (!touching) return

        unbindGlobalEvents()

        //如果追踪列表中没有任何移动事件，那么不会触发后续动作，只是重置数据
        if (!trackList.some((t) => t.type === 'move')) {
            trackList = []
        } else {
            const coords = getCurrentCoordinate(event)
            trackList.push({
                x: coords.x - touching.x,
                y: coords.y - touching.y,
                type: 'up',
                t: Date.now() - touching.startTime,
            })

            unbindTriggerEvents()

            onVerify({
                startSlidingTime: new Date(touching.startTime).toISOString(),
                endSlidingTime: new Date().toISOString(),
                trackList,
            })
        }

        touching = null
    }

    /**
     * 解除触发元素上的事件
     */
    const unbindTriggerEvents = () => {
        triggers.forEach((el) => {
            el.removeEventListener('touchstart', onDragStart)
            el.removeEventListener('mousedown', onDragStart)
        })
    }

    /**
     * 解除全局元素上的事件
     */
    const unbindGlobalEvents = () => {
        window.removeEventListener('touchmove', onDragMove)
        window.removeEventListener('mousemove', onDragMove)
        window.removeEventListener('touchend', onDragEnd)
        window.removeEventListener('mouseup', onDragEnd)
    }

    //如果传入了触发元素则自动绑定事件
    triggers.forEach((el) => {
        el.addEventListener('touchstart', onDragStart, { passive: true })
        el.addEventListener('mousedown', onDragStart, { passive: true })
    })

    return {
        destroy: () => {
            unbindTriggerEvents()
            unbindGlobalEvents()
        },
        onDragStart,
    }
}

/**
 * 创建拖动填充型验证码的配置项
 */
export interface CreateSliderCaptchaOptions
    extends Omit<CreateCaptchaOptions, 'max'> {
    /**
     * 实际显示尺寸与原始图片的比例
     */
    ratio: number
}

/**
 * 创建拖动填充型验证码实例
 * @param options
 */
export function createSliderCaptcha(
    options: CreateSliderCaptchaOptions,
): CaptchaInstance {
    const { captcha, ratio, onVerify, ...rest } = options

    const max = Math.round(
        (captcha.backgroundImageWidth - captcha.templateImageWidth) * ratio,
    )

    return createCaptcha({
        ...rest,
        captcha,
        max,
        onVerify: (data) => {
            onVerify({
                ...data,
                bgImageWidth: Math.round(captcha.backgroundImageWidth * ratio),
                bgImageHeight: Math.round(
                    captcha.backgroundImageHeight * ratio,
                ),
                sliderImageWidth: Math.round(
                    captcha.templateImageWidth * ratio,
                ),
                sliderImageHeight: Math.round(
                    captcha.templateImageHeight * ratio,
                ),
            })
        },
    })
}

/**
 * 创建旋转型验证码的配置项
 */
export interface CreateRotateCaptchaOptions
    extends Omit<CreateCaptchaOptions, 'onDrag'> {
    /**
     * 实际显示尺寸与原始图片的比例
     */
    ratio: number
    /**
     * 当进行拖动时触发的回调
     * @param x 拖动的距离
     * @param deg 旋转的角度
     */
    onDrag: (x: number, deg: number) => void
}

/**
 * 创建旋转型验证码实例
 */
export function createRotateCaptcha(
    options: CreateRotateCaptchaOptions,
): CaptchaInstance {
    const { captcha, ratio, onVerify, onDrag, max, ...rest } = options
    return createCaptcha({
        ...rest,
        captcha,
        max,
        onDrag: (x) => onDrag(x, (x / max) * 360),
        onVerify: (data) => {
            onVerify({
                ...data,
                bgImageWidth: max,
                bgImageHeight: Math.round(
                    captcha.backgroundImageHeight * ratio,
                ),
                sliderImageWidth: Math.round(
                    captcha.templateImageWidth * ratio,
                ),
                sliderImageHeight: Math.round(
                    captcha.templateImageHeight * ratio,
                ),
            })
        },
    })
}

/**
 * 创建上下拼接型验证码的配置项
 */
export interface CreateConcatCaptchaOptions extends CreateCaptchaOptions {
    /**
     * 实际显示尺寸与原始图片的比例
     */
    ratio: number
}

/**
 * 获取上下拼接型验证码的滑块高度
 */
export function getConcatTrackHeight(
    captcha: CaptchaDisplayData,
    ratio: number,
) {
    return Math.round(
        (captcha.backgroundImageHeight - captcha.data.randomY) * ratio,
    )
}

/**
 * 创建上下拼接型验证码实例
 */
export function createConcatCaptcha(
    options: CreateConcatCaptchaOptions,
): CaptchaInstance {
    const { captcha, ratio, onVerify, max, ...rest } = options
    return createCaptcha({
        ...rest,
        captcha,
        max,
        onVerify: (data) => {
            onVerify({
                ...data,
                bgImageWidth: Math.round(captcha.backgroundImageWidth * ratio),
                bgImageHeight: Math.round(
                    captcha.backgroundImageHeight * ratio,
                ),
                sliderImageWidth: Math.round(
                    captcha.backgroundImageWidth * ratio,
                ),
                sliderImageHeight: getConcatTrackHeight(captcha, ratio),
            })
        },
    })
}
