import en from './en.json'
import zhCn from './zh-cn.json'
import zhTw from './zh-tw.json'

/**
 * 可用的翻译数据
 */
const translations: Record<string, Record<string, string>> = {
    en,
    'zh-cn': zhCn,
    'zh-tw': zhTw,
}

/**
 * 可用的语言标识
 */
const availableLanguages = Object.keys(translations)

/**
 * 默认的语言标识
 */
const DEFAULT_LANGUAGE = 'zh-cn'

/**
 * 标准化语言标识
 * @param language
 */
function normalizeLanguage(language: string): string {
    const parts = language.split(/_|-/)
    if (parts.length > 2) {
        parts.splice(1, parts.length - 2)
    }
    return parts.map((t) => t.toLowerCase()).join('-')
}

/**
 * 根据输入值，返回可用的语言
 * @param language
 */
export function getLanguage(language?: string): string {
    let languages =
        typeof language === 'string' && language
            ? [language]
            : navigator && navigator.languages
            ? navigator.languages
            : [DEFAULT_LANGUAGE]

    languages = languages.map((t) => normalizeLanguage(t))

    for (let language of languages) {
        //如果是英文那就返回英文
        if (language === 'en' || language.startsWith('en-')) {
            return 'en'
        }

        //如果是中文那就判断是简体还是繁体
        if (availableLanguages.includes(language)) {
            return language
        }

        //都不是就继续循环
    }

    //到这里表示既不是英文，也不包含任何中文标识，那么判断标识里有没有中文，有就用简体中文，没有就用英文
    if (languages.some((t) => t === 'zh' || t.startsWith('zh-'))) {
        return 'zh-cn'
    }

    return 'en'
}

/**
 * 获取翻译字典
 * @param language
 */
export function getTranslations(language?: string): Record<string, string> {
    return translations[getLanguage(language)] ?? {}
}

/**
 * 获取翻译函数
 * @param language
 */
export function getTranslator(language?: string): (id: string) => string {
    const translation = getTranslations(language)
    return (id: string) => translation[id] ?? id
}
