import { ApiResp, CaptchaData } from './types'

/**
 * HTTP请求配置项
 */
export interface HttpRequestOptions {
    /**
     * 请求地址
     */
    url: string
    /**
     * 基地址
     */
    baseURL?: string
    /**
     * 请求方法
     */
    method?: string
    /**
     * 请求参数
     */
    params?: Record<string, string>
    /**
     * 请求头
     */
    headers?: Record<string, string>
    /**
     * 请求体
     */
    data?: any
    /**
     * 请求超时时间
     */
    timeout?: number
    /**
     * 跨域时是否包含身份信息
     */
    withCredentials?: boolean
}

export interface HttpResponse<T = any> {
    /**
     * 响应码
     */
    status: number
    /**
     * 响应头
     */
    headers: Record<string, string>
    /**
     * 响应体
     */
    data: T
}

function getURL(
    url: string,
    baseURL?: string,
    params?: Record<string, string>,
): string {
    if (!baseURL && !params) return url
    const uri = new URL(url, baseURL)
    if (params) {
        Object.entries(params).forEach(([name, value]) =>
            uri.searchParams.append(name, value),
        )
    }
    return uri.href
}

function parseResponseHeaders(xhr: XMLHttpRequest): Record<string, string> {
    const headers: Record<string, string> = {}

    xhr.getAllResponseHeaders()
        .split('\n')
        .forEach((line) => {
            line = line.trim()
            if (line === '') return
            let [name] = line.split(':')
            name = name.trim()
            if (name === '') return
            const value = xhr.getResponseHeader(name)
            if (value === null) return
            headers[name] = value
        })

    return headers
}

/**
 * 执行HTTP请求
 * @param options 请求配置项
 * @returns
 */
function request<T = any>(
    options: HttpRequestOptions,
): Promise<HttpResponse<T>> {
    const {
        url,
        baseURL,
        method,
        params,
        headers,
        data,
        timeout,
        withCredentials = false,
    } = options

    const xhr = new XMLHttpRequest()
    xhr.responseType = 'text'
    xhr.withCredentials = withCredentials
    if (typeof timeout === 'number') {
        xhr.timeout = timeout
    }

    return new Promise((resolve, reject) => {
        xhr.addEventListener('abort', reject)
        xhr.addEventListener('error', reject)
        xhr.addEventListener('timeout', reject)
        xhr.addEventListener('load', function () {
            resolve({
                status: xhr.status,
                headers: parseResponseHeaders(xhr),
                data: JSON.parse(xhr.responseText),
            })
        })
        xhr.addEventListener('readystatechange', function () {
            if (xhr.readyState !== XMLHttpRequest.OPENED) return
            if (typeof headers === 'object' && headers) {
                Object.entries(headers).forEach(([name, value]) =>
                    xhr.setRequestHeader(name, value),
                )
            }
            if (typeof data !== 'undefined' && data !== null) {
                if (typeof data === 'string') {
                    xhr.send(data)
                    return
                } else if (typeof data === 'object') {
                    xhr.setRequestHeader('Content-Type', 'application/json')
                    xhr.send(JSON.stringify(data))
                    return
                }
            }
            xhr.send()
        })
        xhr.open(method ?? 'GET', getURL(url, baseURL, params))
    })
}

/**
 * 加载验证码数据
 * @param options 获取验证码数据的配置项
 */
export async function loadCaptcha(options: HttpRequestOptions) {
    while (true) {
        const resp = await request<ApiResp<CaptchaData>>({
            method: 'POST',
            data: {},
            ...options,
        })
        if (
            ['SLIDER', 'ROTATE', 'CONCAT'].includes(resp.data.data.captcha.type)
        ) {
            return resp.data
        }
    }
}

export interface ValidCaptchaOptions
    extends Omit<HttpRequestOptions, 'method' | 'data'> {
    data: {
        key: string
        data: any
    }
}

/**
 * 校验验证码数据，校验成功返回undefined，校验失败返回失败文字
 * @param options
 */
export async function validCaptcha(
    options: ValidCaptchaOptions,
): Promise<string | void> {
    const { data, ...rest } = options

    let result: ApiResp<{
        key: string
    }>

    try {
        const resp = await request({
            ...rest,
            method: 'POST',
            data,
        })
        result = resp.data
    } catch {
        //发生了请求异常
        return '验证失败，请再试一次'
    }

    if (result.code !== 0) {
        return result.code === 4000 ? '验证码已过期' : '验证失败，请再试一次'
    }
}
