import en from './en.json'
import zhCn from './zh-cn.json'
import zhTw from './zh-tw.json'

/**
 * 可用的翻译数据
 */
const translations: Record<string, Record<string, string>> = {
    en,
    'zh-cn': zhCn,
    'zh-tw': zhTw,
}

/**
 * 默认的语言标识
 */
const DEFAULT_LANGUAGE = 'zh-cn'

/**
 * 标准化语言标识
 * @param language
 */
function normalizeLanguage(language: string) {
    const parts = language.toLowerCase().split(/_|-/)
    const lang = parts[0]
    const country = parts.length > 1 ? parts[parts.length - 1] : undefined
    const variant = parts.length > 2 ? parts[1] : undefined
    return {
        lang,
        country,
        variant,
    }
}

/**
 * 根据输入值，返回可用的语言
 * @param language
 */
export function getLanguage(language?: string): string {
    let languages =
        typeof language === 'string' && language
            ? [language]
            : navigator && navigator.languages
            ? navigator.languages
            : [DEFAULT_LANGUAGE]

    const langs = languages.map((t) => normalizeLanguage(t))

    for (let lang of langs) {
        //如果是英文那就返回英文
        if (lang.lang === 'en') return 'en'
        if (lang.lang === 'zh') {
            //有变体优先判断变体
            if (lang.variant === 'hans') {
                return 'zh-cn'
            } else if (lang.variant === 'hant') {
                return 'zh-tw'
            }

            //没有变体判断国家标识
            if (lang.country === 'cn') {
                return 'zh-cn'
            } else if (lang.country === 'tw') {
                return 'zh-tw'
            }

            //没有变体也没有国家标识就用简体中文
            return 'zh-cn'
        }
    }

    //到这里表示既不是英文，也不包含任何中文标识，那就用英文
    return 'en'
}

/**
 * 获取翻译字典
 * @param language
 */
export function getTranslations(language?: string): Record<string, string> {
    return translations[getLanguage(language)] ?? {}
}

/**
 * 获取翻译函数
 * @param language
 */
export function getTranslator(language?: string): (id: string) => string {
    const translation = getTranslations(language)
    return (id: string) => translation[id] ?? id
}
