import { md5 } from '../utils/security'
import { InternalApiAxiosRequestConfig } from './types'

/**
 * 请求头中参与签名的字段
 */
const HEADER_KEYS = [
    'source',
    'device',
    'app_version',
    'token',
    'partner_key',
    'sdk_version',
    'timestamp',
    'request_number',
]

/**
 * 注入请求签名
 * @param config
 */
export default function injectSignature(
    config: InternalApiAxiosRequestConfig,
): InternalApiAxiosRequestConfig {
    //先获取签名密钥，如果没有密钥则不签名
    const { signature_secret } = config
    if (typeof signature_secret !== 'string' || signature_secret === '') return config

    /**
     * 待签名的数据
     */
    const data = {
        ...config.data,
    }

    //取出请求头中参与签名的字段
    HEADER_KEYS.forEach((key) => {
        if (config.headers.has(key)) {
            data[key] = config.headers.get(key)
        }
    })

    config.headers.set('signature', createSignature(data, signature_secret))

    return config
}

/**
 * 计算签名
 * @param data
 * @param secret
 */
export function createSignature(data: Record<string, any>, secret: string): string {
    //先对参数名按字典序排个序
    const entries = Object.entries(data).sort((t1, t2) => t1[0].localeCompare(t2[0]))
    //再根据参数值，生成对应的字符串值
    const values = entries.map(([_, value]) => {
        if (
            typeof value === 'undefined' ||
            typeof value === 'function' ||
            typeof value === 'symbol' ||
            value === null
        ) {
            return ''
        } else if (typeof value === 'object') {
            return JSON.stringify(value) ?? ''
        } else {
            return String(value)
        }
    })

    //拼接原始字符串值，并在后面追加签名密钥，得到基础签名串
    const baseStr = values.join('') + secret

    //进行2次md5，得到最终签名串
    return md5(md5(baseStr))
}
