import {
    AxiosRequestHeaders,
    InternalAxiosRequestConfig,
    type AxiosInstance,
} from 'axios'
import { sample } from 'lodash-es'
import {
    aesDecrypt,
    aesEncrypt,
    base64Decode,
    base64Encode,
    md5,
} from './security'

/**
 * 接口调用配置项
 */
export interface ApiOptions {
    /**
     * 接口请求的基地址
     */
    baseURL?: string
    /**
     * 包名
     */
    source?: string
    /**
     * 设备类型
     */
    device?: string
    /**
     * 版本号
     */
    appversion?: string
    /**
     * 附带在请求头的token或者token获取函数
     */
    token?: string | (() => string | undefined | null)
    /**
     * 请求体是否加密
     */
    encryption?: boolean
}

/**
 * 请求加解密密钥
 */
export interface SignKey {
    name: string
    key: string
    iv: string
}

/**
 * 默认的请求加解密密钥列表
 */
export const DEFAULT_SIGN_KEYS: SignKey[] = [
    {
        key: 'k0f3JfxEWd3HC7pXQU8tmSkDheUXibmz',
        iv: 'Fu2wU73MBcsEZWJk',
        name: 'secret_key_1',
    },
    {
        key: 'NdEHDuAZGCQV6C0oaURvBJJWA4z2QHyG',
        iv: 'w0j9K4bswcGJKtj5',
        name: 'secret_key_2',
    },
    {
        key: 'lBsn3FrYN9hdoFqMdqqNV1dlpGutkcXk',
        iv: 'NRHqJYTNKxH8Z9fU',
        name: 'secret_key_3',
    },
    {
        key: 'eVdEc44rke6P6rfn9SgEWGPNqkcipWN7',
        iv: 'l5HNU6Q0bdc2piB3',
        name: 'secret_key_4',
    },
    {
        key: '53BSHwE29I1u4e5cJDMZGHUdi0A7L4E5',
        iv: 'zUsjrk58p2RahP08',
        name: 'secret_key_5',
    },
    {
        key: 'vgizOWReMzVJA6LEsb9N36LEzPqcFdeO',
        iv: 'e52YLbnvVv4HpGu7',
        name: 'secret_key_6',
    },
]

export const DEFAULT_API_OPTIONS: ApiOptions = {
    source: 'xinxiuweb',
    device: 'wap',
    appversion: '1.0.0',
}

function checkHeader(
    headers: AxiosRequestHeaders,
    key: string,
    ...defaultValues: any[]
): void {
    if (headers.has(key)) return
    for (const value of defaultValues) {
        if (typeof value === 'undefined' || value === null) continue
        if (typeof value === 'function') {
            const headerValue = value()
            if (typeof headerValue === 'undefined' || headerValue === null)
                continue
            headers.set(key, headerValue)
        }
        headers.set(key, String(value))
    }
}

/**
 * 为axios实例注入追加公参的拦截器
 * @param axios axios实例
 * @param options 默认配置
 */
export function useCommonParamsInterceptor(
    axios: AxiosInstance,
    options: ApiOptions = {}
) {
    options = {
        ...DEFAULT_API_OPTIONS,
        ...options,
    }

    axios.interceptors.request.use(
        (request: InternalAxiosRequestConfig<any> & ApiOptions) => {
            //包名
            checkHeader(
                request.headers,
                'source',
                request.source,
                options.source
            )
            //设备类型
            checkHeader(
                request.headers,
                'device',
                request.device,
                options.device
            )
            //版本号
            checkHeader(
                request.headers,
                'appversion',
                request.appversion,
                options.appversion
            )
            //时间戳
            checkHeader(
                request.headers,
                'timestamp',
                Math.floor(Date.now() / 1000)
            )
            //请求标识
            checkHeader(
                request.headers,
                'request_number',
                md5(String(Date.now() + Math.random()))
            )

            return request
        }
    )
}

function getParamValue(value: any): string {
    if (value === null || typeof value === 'undefined') {
        return ''
    }
    if (typeof value === 'object') {
        return JSON.stringify(value)
    }
    return String(value)
}

/**
 * 为axios实例注入请求签名拦截器
 * @param axios axios实例
 * @param key 签名密钥，如果不传则使用默认密钥
 */
export function useSignatureInterceptor(axios: AxiosInstance, key?: string) {
    axios.interceptors.request.use((request) => {
        if (request.headers.has('signature')) return request

        const body =
            typeof request.data === 'object' &&
            request.data !== null &&
            !Array.isArray(request.data)
                ? request.data
                : {}

        //参与签名的数据
        const params: Record<string, any> = {
            ...body,
            source: request.headers.get('source'),
            device: request.headers.get('device'),
            timestamp: request.headers.get('timestamp'),
            request_number: request.headers.get('request_number'),
            appversion: request.headers.get('appversion'),
        }
        const entries = Object.entries(params).sort((k1, k2) =>
            k1[0].localeCompare(k2[0])
        )

        let basestr = entries.map((t) => getParamValue(t[1])).join('')
        if (key) {
            basestr += key
        } else if (request.headers.get('device') === 'wap') {
            basestr += 'asdasgfdwqew'
        } else {
            basestr += 'WQdflHhKjdafsj21321jkl531l45'
        }
        basestr = md5(md5(basestr))

        const randomStr = md5(String(Date.now() + Math.random()))
        const signature =
            randomStr.substring(0, 5) + basestr + randomStr.substring(6, 12)

        request.headers.set('signature', signature)

        return request
    })
}

/**
 * 为axios实例注入请求加解密拦截器
 * @param axios axios实例
 * @param signKeys 密钥列表，如果不传则使用默认密钥
 */
export function useEncryptInterceptor(
    axios: AxiosInstance,
    signKeys: SignKey[] = DEFAULT_SIGN_KEYS
) {
    axios.interceptors.request.use(
        (request: InternalAxiosRequestConfig<any> & ApiOptions) => {
            if (request.encryption === false) return request
            let data = request.data
            const strData =
                typeof data === 'string'
                    ? data
                    : typeof data === 'undefined' || data === null
                    ? '{}'
                    : JSON.stringify(data)
            let sign_name = request.headers.get('sign_name') as string
            let signKey: SignKey
            if (sign_name) {
                signKey = signKeys.find((item) => item.name === sign_name)!
                if (!signKey) {
                    signKey = sample(signKeys)!
                }
            } else {
                signKey = sample(signKeys)!
            }
            const { key, name, iv } = signKey

            request.headers.set('sign_name', name)

            const encrypted = aesEncrypt(strData, key, iv)
            const base64 = base64Encode(encrypted)
            const sign_data = encodeURIComponent(base64)

            request.data = { sign_data }

            return request
        }
    )

    axios.interceptors.response.use((resp) => {
        const sign_name = resp.config.headers.get('sign_name')
        if (!sign_name || typeof resp.data !== 'object' || resp.data === null) {
            return resp
        }

        const signKey = signKeys.find((t) => t.name === sign_name)
        if (!signKey) return resp

        if (typeof resp.data.sign_data !== 'string' || !resp.data.sign_data) {
            return resp
        }

        let signData = resp.data.sign_data
        signData = decodeURIComponent(signData)
        signData = base64Decode(signData)
        signData = aesDecrypt(signData, signKey.key, signKey.iv)
        resp.data = JSON.parse(signData)
        return resp
    })
}
