interface IosWebkitInterface {
    messageHandlers: {
        [name: string]: {
            postMessage(data?: any): void
        }
    }
}

declare global {
    interface Window {
        webkit: IosWebkitInterface

        liveapp: {
            [name: string]: (data?: string) => void
        }
    }
}

function callIosApi<T>(name: string, data: any, ack?: (data: T) => void): void
function callIosApi<T>(name: string, ack?: (data: T) => void): void
function callIosApi<T>(name: string, ...args: any[]): void {
    if (typeof window.webkit?.messageHandlers[name]?.postMessage === 'function') {
        return
    }

    let params: any = undefined
    let ack: Function | undefined = undefined
    if (typeof args[0] === 'function') {
        ack = args[0]
    } else if (typeof args[1] === 'function') {
        params = args[0]
        ack = args[1]
    }

    if (typeof ack === 'function') {
        // @ts-ignore
        window[name] = ack
    }

    window.webkit.messageHandlers[name].postMessage(params ?? true)
}

function callAndroidApi<T>(name: string, data: any, ack?: (data: T) => void): void
function callAndroidApi<T>(name: string, ack?: (data: T) => void): void
function callAndroidApi(name: string, ...args: any[]): void {
    if (typeof window.liveapp[name] !== 'function') {
        return
    }

    let params: any = undefined
    let ack: Function | undefined = undefined
    if (typeof args[0] === 'function') {
        ack = args[0]
    } else if (typeof args[1] === 'function') {
        params = args[0]
        ack = args[1]
    }

    let result: any
    if (typeof params !== 'undefined' && params !== null) {
        result = window.liveapp[name](JSON.stringify(params))
    } else {
        result = window.liveapp[name]()
    }
    if (typeof ack === 'function') {
        ack(result)
    }
}

const isiOS = !!navigator.userAgent.match(/\(i[^;]+;( U;)? CPU.+Mac OS X/) //ios终端

/**
 * 调用原生提供的api
 * @param name
 * @param params
 */
export function callNativeApi<T>(name: string, data: any, ack?: (data: T) => void): void
export function callNativeApi<T>(name: string, ack?: (data: T) => void): void
export function callNativeApi(name: string, ...args: any[]): void {
    isiOS ? callIosApi(name, ...args) : callAndroidApi(name, ...args)
}

export namespace NativeApi {
    interface ActionAsPhpData {
        app_open: string
        url: string
        ios_url: string
        android_url: string
    }

    /**
     * 跳转到
     * @param data 跳转参数
     */
    export function actionAsPhp(data: ActionAsPhpData): void {
        callNativeApi('actionAsPhp', data)
    }

    /**
     * 关闭当前页面
     */
    export function closeWeb() {
        callNativeApi('closeWeb')
    }

    /**
     * 获取状态栏高度
     * @returns
     */
    export function getStatusBarHeight(): Promise<number> {
        return new Promise<number>((resolve) => callNativeApi('getStatusBarHeight', resolve))
    }

    /**
     * 获取底部安全区高度
     * @returns
     */
    export function getBottomSafeHeight(): Promise<number> {
        return new Promise<number>((resolve) => callNativeApi('getBottomSafeHeight', resolve))
    }

    /**
     * 绑定支付宝
     */
    export function openBindZfb() {
        callNativeApi('openBindZfb')
    }

    /**
     * 打开银行列表
     */
    export function openBankList() {
        callNativeApi('openBankList')
    }

    interface ShareTipData {
        url: string
        img: string
        title: string
        desc: string
    }

    /**
     * 分享
     * @param data
     */
    export function openShareTip(data: ShareTipData) {
        callNativeApi('openShareTip', data)
    }

    /**
     * 打开个人中心
     * @param userId
     */
    export function toPerson(userId: number | string) {
        callNativeApi('toPerson', { userid: String(userId) })
    }

    /**
     * 验证⽤户身份
     * @param action 1-进入手机认证 2-进入实名认证
     */
    export function verify(action: '1' | '2') {
        callNativeApi('verify', { action })
    }

    /**
     * 进入直播间
     * @param userId
     * @param avatar
     */
    export function toLiveroom(userId: number | string, avatar?: string) {
        callNativeApi('toLiveroom', { userid: String(userId), avatar })
    }

    /**
     * 清除浏览器缓存
     */
    export function removeHistory() {
        callNativeApi('removeHistory')
    }

    /**
     * 去开播
     */
    export function openLive() {
        callNativeApi('openLive')
    }
}
