import CryptoJS from 'crypto-js'
import Cookies from 'js-cookie'

/**
 * token数据
 */
export interface TokenData {
    /**
     * token的来源
     */
    source: 'hash_raw' | 'hash_encoded' | 'url_raw' | 'url_encoded' | 'cookie_app' | 'cookie'
    /**
     * token值
     */
    token: string
}

/**
 * 解码token使用的key
 */
const key = CryptoJS.enc.Utf8.parse('cqfjkjgs')

/**
 * 解码token
 */
function decodeToken(token: string) {
    return CryptoJS.AES.decrypt(token, key, {
        mode: CryptoJS.mode.ECB,
        padding: CryptoJS.pad.Pkcs7,
    }).toString(CryptoJS.enc.Utf8)
}

/**
 * 尝试解析token，按以下顺序解析
 * 1.地址上的加密token
 * 2.地址上的明文token
 * 3.cookie里的app_token
 * 4.cookie里的token
 */
export function parseToken(): TokenData | undefined {
    const url = new URL(location.href)
    if (url.hash && url.hash.includes('?')) {
        const hashParams = new URLSearchParams(url.hash.substring(url.hash.indexOf('?') + 1))
        //hash上有token
        if (hashParams.has('token')) {
            const token = hashParams.get('token')
            if (token) {
                //先尝试解码，如果失败则使用明文
                try {
                    return {
                        token: decodeToken(token),
                        source: 'hash_encoded',
                    }
                } catch (e: unknown) {
                    return {
                        token: decodeToken(token),
                        source: 'hash_raw',
                    }
                }
            }
        }
    }

    //然后尝试从url上获取token
    if (url.searchParams.has('token')) {
        const token = url.searchParams.get('token')
        if (token) {
            //先尝试解码，如果失败则使用明文
            try {
                return {
                    token: decodeToken(token),
                    source: 'url_encoded',
                }
            } catch (e: unknown) {
                return {
                    token: decodeToken(token),
                    source: 'url_raw',
                }
            }
        }
    }

    //接着尝试从cookie中获取app_token
    const appToken = Cookies.get('app_token')
    if (appToken) {
        return {
            token: appToken,
            source: 'cookie_app',
        }
    }

    //再尝试从cookie中获取token
    const token = Cookies.get('token')
    if (token) {
        return {
            token,
            source: 'cookie',
        }
    }
}
