import axios, {
    AxiosInstance,
    AxiosInterceptorManager,
    AxiosPromise,
    CreateAxiosDefaults,
} from 'axios'
import decryptResponse from './decrypt-response'
import encryptRequest from './encrypt-request'
import injectCommonParams from './inject-common-params'
import injectSignature from './inject-signature'
import {
    ApiAxiosRequestConfig,
    ApiAxiosResponse,
    ApiRequestData,
    ApiRequestDefaultOptions,
    GetterOrValue,
    InternalApiAxiosRequestConfig,
} from './types'
import { sample } from 'lodash-es'

/**
 * 用于执行接口请求的，拥有特殊选项的Axios请求实例
 */
export interface ApiAxiosInstance {
    <TData = any, TExtend extends object = {}>(config: ApiAxiosRequestConfig): Promise<
        ApiAxiosResponse<TData, TExtend>
    >

    request<TData = any, TExtend extends object = {}>(
        config: ApiAxiosRequestConfig,
    ): Promise<ApiAxiosResponse<TData, TExtend>>

    /**
     * 设置在实例上的默认配置项
     */
    defaults: AxiosInstance['defaults'] & ApiRequestDefaultOptions
    /**
     * 拦截器
     */
    interceptors: {
        /**
         * 请求拦截器
         */
        request: AxiosInterceptorManager<InternalApiAxiosRequestConfig>
        /**
         * 响应拦截器
         */
        response: AxiosInterceptorManager<ApiAxiosResponse>
    }
}

export interface CreateInstanceDefaults
    extends CreateAxiosDefaults<ApiRequestData>,
        ApiRequestDefaultOptions {}

/**
 * 可以被简单处理的内置请求配置项
 */
const INTERNAL_OPTION_KEYS: (keyof ApiRequestDefaultOptions)[] = [
    'source',
    'device',
    'app_version',
    'token',
    'partner_key',
    'sdk_version',
    'signature_secret',
]

/**
 * 创建用于请求接口的Axios实例
 */
export function createAxiosInstance(
    options: CreateInstanceDefaults = {
        source: 'xinxiuweb',
        app_version: '1.0.0',
        device: 'wap',
        signature_secret: 'asdasgfdwqew',
    },
): ApiAxiosInstance {
    const instance = axios.create(options)

    //按倒序注入请求拦截器，最后处理的最先注入
    //加密请求体
    instance.interceptors.request.use(encryptRequest)

    //计算签名并注入
    instance.interceptors.request.use(injectSignature)

    //自动填充公参
    instance.interceptors.request.use(injectCommonParams)

    //响应拦截器，解密响应体
    instance.interceptors.response.use(decryptResponse)

    /**
     * 实际的接口调用方法
     * @param config
     */
    const request = async (config: ApiAxiosRequestConfig): AxiosPromise => {
        const new_config = { ...config }

        //先对配置项进行初始化，把默认配置项里的函数型配置项，转换为具体的值
        //可以被自动转换的配置
        for (const key of INTERNAL_OPTION_KEYS) {
            new_config[key] = await getValues<any>(
                config[key],
                (instance.defaults as ApiRequestDefaultOptions)[key],
            )
        }

        //特殊的加解密密钥配置
        const encryption_key = await getValues(
            config.encryption_key,
            (instance.defaults as ApiRequestDefaultOptions).encryption_key,
        )
        if (Array.isArray(encryption_key)) {
            new_config.encryption_key = sample(encryption_key)
        } else {
            new_config.encryption_key = encryption_key
        }

        //最后调用原始axios实例上的方法
        return instance.request(new_config)
    }

    Object.defineProperties(request, {
        defaults: {
            value: instance.defaults,
            writable: false,
        },
        interceptors: {
            value: instance.interceptors,
            writable: false,
        },
        request: {
            value: request,
            writable: false,
        },
    })

    return request as unknown as ApiAxiosInstance
}

/**
 * 读取一个配置项的值，传入的可以是具体的值，也可以是获取的方法
 * @param getterOrValue
 */
async function getValue<T>(getterOrValue: GetterOrValue<T> | undefined): Promise<T | undefined> {
    return typeof getterOrValue === 'function' ? (getterOrValue as () => T)() : getterOrValue
}

/**
 * 读取多个配置项的值，返回找到的第一个有效的配置项，传入的可以是具体的值，也可以是获取的方法
 * @param values
 * @returns
 */
async function getValues<T>(...values: (GetterOrValue<T> | undefined)[]): Promise<T | undefined> {
    for (const getterOrValue of values) {
        const value = await getValue(getterOrValue)
        if (typeof value !== 'undefined') return value
    }
}
