import {
    AxiosInstance,
    AxiosInterceptorManager,
    AxiosRequestConfig,
    AxiosResponse,
    InternalAxiosRequestConfig,
} from 'axios'

/**
 * 接口传递给服务端的数据类型
 */
export type ApiRequestData = Record<string, any>

/**
 * 接口响应基础数据结构
 */
export interface ApiRespBase<TData = any> {
    /**
     * 接口响应码，200表示请求成功，非200表示请求失败
     */
    code: number
    /**
     * 错误消息
     */
    msg: string
    /**
     * 响应数据
     */
    data: TData
}

/**
 * 接口响应完整数据结构
 */
export type ApiResp<TData = any, TExtend extends object = {}> = TExtend & ApiRespBase<TData>

/**
 * 接口请求数据加解密密钥
 */
export interface ApiEncryptionKey {
    /**
     * 密钥名
     */
    name: string
    /**
     * 密钥值
     */
    key: string
    /**
     * 向量
     */
    iv: string
}

/**
 * 接口请求的配置项
 */
export interface ApiRequestOptions {
    /**
     * 请求的包名，如果不设置，默认为`xinxiuweb`
     */
    source?: string
    /**
     * 请求的设备类型，如果不设置，默认为`wap`
     */
    device?: string
    /**
     * 当前客户端的版本号，如果不设置，默认为`1.0.0`
     */
    app_version?: string
    /**
     * 用户的token
     */
    token?: string
    /**
     * 渠道标识
     */
    partner_key?: string
    /**
     * 渠道SDK版本号
     */
    sdk_version?: string
    /**
     * 用于计算签名的密钥，如果不设置，默认为`asdasgfdwqew`
     */
    signature_secret?: string
    /**
     * 接口数据加解密密钥
     */
    encryption_key?: ApiEncryptionKey
}

export type GetterOrValue<T> = T | (() => T | Promise<T>)

/**
 * 接口请求的默认配置项
 */
export type ApiRequestDefaultOptions = {
    [K in keyof ApiRequestOptions]?: GetterOrValue<ApiRequestOptions[K]>
} & {
    /**
     * 接口数据加解密密钥
     */
    encryption_key?: GetterOrValue<ApiEncryptionKey | ApiEncryptionKey[] | undefined>
}

export type InternalApiAxiosRequestConfig = InternalAxiosRequestConfig<ApiRequestData> &
    ApiRequestOptions

export interface ApiAxiosResponse<TData = any, TExtend extends object = {}>
    extends AxiosResponse<ApiResp<TData, TExtend>, ApiRequestData> {
    config: InternalApiAxiosRequestConfig
}

export interface ApiAxiosRequestConfig
    extends AxiosRequestConfig<ApiRequestData>,
        ApiRequestOptions {
    url: string
}
