import { Http } from "foreign-country-utils";
import requestError from "@/request/requestError";
import encodeHeader from "@/request/usePostCommonParam";
import Secret_key from "@/secret_key/secret_key";
import { Base64 } from "js-base64";
import { decodeData, encodeData } from "@/request/useRequestEncryption";
import { AxiosRequestConfig, InternalAxiosRequestConfig } from "axios";
import usePrint from "@/request/usePrint";

const ENV = import.meta.env;
const baseUrl = ENV.VITE_BASE_URL;
const timeout = 30 * 1000;
const secretLength = Object.keys(Secret_key).length;

// 以下是基础请求，可自行添加加密方式
export const http = new Http(baseUrl, timeout, {}, [
	{
		request: {
			before(config: InternalAxiosRequestConfig) {
				const key = `secret_key_${1 + Math.floor(Math.random() * secretLength)}`;
				const secret = Secret_key[key];
				let tempParams: unknown = {};
				const { method, url = "", params, data } = config;
				if (method === "get") {
					ENV.MODE !== "production" && usePrint(params || {}, method, url);
					if (params) {
						tempParams = params;
						const sign_data = encodeURIComponent(
							Base64.encode(encodeData(typeof params === "string" ? params : JSON.stringify(params), secret))
						);
						config.params = { sign_data: sign_data };
					}
				}
				if (method === "post") {
					ENV.MODE !== "production" && usePrint(data || {}, method, url);
					if (data) {
						tempParams = data;
						const sign_data = encodeURIComponent(
							Base64.encode(encodeData(typeof data === "string" ? data : JSON.stringify(data), secret))
						);
						config.data = { sign_data };
					}
				}
				const enHeader = encodeHeader(key, "", tempParams);
				for (const stringsKey in enHeader) {
					config.headers.set(stringsKey, enHeader[stringsKey]);
				}
				return config;
			}
		},
		response: {
			success(result) {
				const data = result.data;
				if (data && typeof data === "object" && "encrypted_data" in data) {
					const secret = Secret_key[result.config.headers.sign_name];
					if (secret) {
						if (typeof data.encrypted_data === "string" && data.encrypted_data.length === 0) {
							// @ts-ignore
							result.data.data = null;
						} else {
							// @ts-ignore
							result.data.data = JSON.parse(decodeURIComponent(Base64.decode(decodeData(data.encrypted_data as string, secret))));
						}
					} else {
						throw "解密失败";
					}
				}
				return result;
			},
			error(err) {
				// 请求错误提醒
				requestError(err.message);
				return Promise.reject(err);
			}
		}
	}
]);

export const get = <T, D = DataObj>(url: string, params?: D, config?: AxiosRequestConfig<D>): Promise<BaseResponse<T>> =>
	http.get(url, params, config);
export const post = <T, D = DataObj>(url: string, data?: D, config?: AxiosRequestConfig<D>): Promise<BaseResponse<T>> =>
	http.post<BaseResponse<T>, D>(url, data, config);
export const upload = <T>(url: string, data: FormData, config?: AxiosRequestConfig): Promise<BaseResponse<T>> =>
	http.upload<BaseResponse<T>>(url, data, config);
export const remove = <T, D = DataObj>(url: string, params?: D, config?: AxiosRequestConfig<D>) =>
	http.delete<BaseResponse<T>, D>(url, params, config);
export const put = <T, D = DataObj>(url: string, params?: D, config?: AxiosRequestConfig<D>) => http.put<BaseResponse<T>, D>(url, params, config);
