<?php

namespace app\api\controller\manage;

use app\api\middleware\Auth;
use app\api\service\UtilService;
use app\BaseController;
use app\model\Course as courseModel;
use app\model\Payment;
use app\model\ProjectTag;
use think\facade\Db;
use think\Request;
use app\model\project as projectModel;
use app\model\project\User as userModel;

class Index extends BaseController
{

    protected $middleware = [
        Auth::class,
    ];

    //看板统计
    public function index(Request $request)
    {
        $userId = $request->userId;

        try {
            Db::startTrans();

            // 1. 获取基础数据
            $courseStats = Db::name('course')
                ->where('user_id', $userId)
                ->field([
                    'COUNT(id) as course_count',
                    'SUM(tvclick) as pv_count',
                    'GROUP_CONCAT(id) as course_ids'
                ])
                ->find();

            $courseIds = $courseStats['course_ids'] ? explode(',', $courseStats['course_ids']) : [];

            // 2. 获取支付相关统计数据
            $paymentStats = Db::name('payment')
                ->where('order_id', 'in', $courseIds)
//                ->where('sf_id', 'in', $courseIds)

                ->where('order_type', 1)
                ->field([
                    'SUM(CASE WHEN pay_status = 1 THEN pay_amount ELSE 0 END) as order_money_sum',
                    'COUNT(id) as order_count'
                ])
                ->find();

            // 3. 获取项目统计数据
            $projectStats = Db::name('project')
                ->where('user_id', $userId)
                ->field([
                    'COUNT(id) as project_count',
                    'SUM(CASE WHEN status IN (0,1,3) THEN 1 ELSE 0 END) as project_pending_count',
                    'SUM(CASE WHEN status = 2 THEN 1 ELSE 0 END) as project_progress_count',
                    'SUM(CASE WHEN status = 4 THEN 1 ELSE 0 END) as project_finished_count'
                ])
                ->find();

            // 组装返回数据
            $result = [
                'orderMoneySum' => (int)$paymentStats['order_money_sum'] ?? 0,
                'orderCount' => (int)$paymentStats['order_count'] ?? 0,
                'courseCount' => (int)$courseStats['course_count'] ?? 0,
                'pvCount' => (int)$courseStats['pv_count'] ?? 0,
                'projectCount' => (int)$projectStats['project_count'] ?? 0,
                'projectPendingCount' => (int)$projectStats['project_pending_count'] ?? 0,
                'projectProgressCount' => (int)$projectStats['project_progress_count'] ?? 0,
                'projectFinishedCount' => (int)$projectStats['project_finished_count'] ?? 0,
//                'projectPendingList'=>$this->getProjectPendingList($userId)
            ];

            Db::commit();

            return $this->returnMsg('success', 1,$result);

        } catch (\Exception $e) {
            Db::rollback();
            return $this->returnMsg('系统错误，请稍后再试');
        }
    }

    public function getProjectPendingList(Request $request)
    {
        $userId = $request->userId;
        $page = $request->param('page/d', 1);
        $pageSize = $request->param('pageSize/d', 10);


        // 1. 查询项目列表
        $projects = \app\model\Project::where('user_id', $userId)
            ->where('status', 'in', [0, 1, 3])
            ->withCount(['put'])
            ->field([
                'id',
                'title',
                'status',
                'tag_ids',
                'sh_status_time',
                'put_time',
                'zhouqi',
                // 当 status=3 时，计算截止日期（put_time + zhouqi * 86400）
                'IF(
                    status=3 AND put_time IS NOT NULL AND zhouqi IS NOT NULL, 
                    FROM_UNIXTIME(put_time + zhouqi * 86460), 
                    NULL
                ) as deadline_date',
            ])
            ->paginate([
                'page' => $page,      // 当前页数
                'list_rows' => $pageSize, // 每页数量
            ]);

        $list = UtilService::listWithTags($projects,ProjectTag::class,'tag_ids');

        return $this->returnMsg('success', 1, $list);
    }

    public function card(Request $request)
    {
        $userId = $request->userId;

        $courseCount = courseModel::where(['user_id'=>$userId,'is_del'=>0,'status'=>3])->count();

        $projectCount = projectModel::where(['user_id'=>$userId,'sh_status'=>2])->count();

        $orderMoneySum = Payment::where(['store_user_id'=>$userId,'order_type'=>1,'pay_status'=>1])->sum('order_price');

        $projectCreditoney = userModel::where('id',$userId)->value('credit_money');

        return $this->returnMsg('success', 1,
            [
            'courseCount'=>$courseCount,
            'projectCount'=>$projectCount,
            'orderMoneySum'=>$orderMoneySum,
            'projectCreditoney'=>$projectCreditoney
            ]
        );

    }

    //图表统计
    public function getStatistics(Request $request)
    {

        $range = $request->param('range', 'month');
        $validRanges = ['month', 'quarter', 'halfyear', 'all','year'];

        if (!in_array($range, $validRanges)) {
            return $this->returnMsg('无效时间范围');
        }
        $userId = $request->userId;


        $payment = new Payment();
        $result = $payment->getOrderStatistics($range,$userId);

        // 格式化数据
        $formattedData = $this->formatStatisticsData($result['data'], $result['format']);


        return $this->returnMsg('success', 1, [
            'data' => $formattedData,
            'time_format' => $result['format']
        ]);


    }

    /**
     * 格式化统计结果
     */
    protected function formatStatisticsData($data, $format)
    {
        $result = [
            'time_periods' => [],
            'order_counts' => [],
            'total_amounts' => []
        ];

        foreach ($data as $item) {
            $result['time_periods'][] = $item['time_period'];
            $result['order_counts'][] = $item['order_count'];
            $result['total_amounts'][] = (float)$item['total_amount'];
        }

        // 如果是按月统计，添加月份显示格式
        if ($format === 'month') {
            $result['display_periods'] = array_map(function($period) {
                return date('Y年m月', strtotime($period.'-01'));
            }, $result['time_periods']);
        } else {
            $result['display_periods'] = array_map(function($period) {
                return date('m月d日', strtotime($period));
            }, $result['time_periods']);
        }

        return $result;
    }

    public function pendingWorkList(Request $request)
    {
        // 获取高校/企业用户ID
        $userId = $request->userId;

        if (empty($userId)) {
            return $this->returnMsg('用户ID不能为空');
        }

        try {
            // 查询该高校下的所有作业
            $works = Db::name('course_work')
                ->alias('zy')
                ->join('course kc','zy.course_id=kc.id')
                ->join('course_category kcfl','kc.cate_id=kcfl.id')
//                ->where('user_id', $userId)
                ->where('zy.is_del', 0)
                ->field('zy.id, zy.title, zy.course_id,kc.title as course_name,kc.cate_id,kcfl.title as course_category')
                ->select()
                ->toArray();

            if (empty($works)) {
                return $this->returnMsg('success', 1,[]);
            }

            $workIds = array_column($works, 'id');

            // 查询每个作业的提交情况
            $submissions = Db::name('course_user_work')
                ->where('work_id', 'in', $workIds)
                ->where('is_del', 0)
                ->field('work_id, status, COUNT(*) as count')
                ->group('work_id, status')
                ->select()
                ->toArray();
            // 组织统计数据
            $stats = [];
            foreach ($submissions as $sub) {
                $workId = $sub['work_id'];
                if (!isset($stats[$workId])) {
                    $stats[$workId] = [
                        'submitted' => 0,  // 已提交人数
                        'pending' => 0      // 待批改人数(已提交但未批改)
                    ];
                }

                if ($sub['status'] == 1) { // 已提交
                    $stats[$workId]['submitted'] += $sub['count'];
                    $stats[$workId]['pending'] += $sub['count'];
                } elseif ($sub['status'] >= 2) { // 已批改(通过或未通过)
                    $stats[$workId]['submitted'] += $sub['count'];
                }
            }

            // 构建返回数据
            $result = [];
            foreach ($works as $work) {
                $workId = $work['id'];
                $result[] = [
                    'work_id' => $workId,
                    'title' => $work['title'],
                    'course_id' => $work['course_id'],
                    'submitted_count' => $stats[$workId]['submitted'] ?? 0,
                    'pending_count' => $stats[$workId]['pending'] ?? 0,
                    'course_name'=>$work['course_name'],
                    'course_category'=>$work['course_category']
                ];
            }
            return $this->returnMsg('success', 1,$result);

        } catch (\Exception $e) {
            return $this->returnMsg($e->getMessage());
        }



    }
}