<?php
namespace app\model;

use app\model\project\User;
use think\Model;
use think\model\concern\SoftDelete;
use think\facade\Db;

class CourseComment extends Model
{
    use SoftDelete;

    // 设置当前模型对应的完整数据表名称
    protected $table = 'fj_course_comments';

    // 自动时间戳
    protected $autoWriteTimestamp = true;
    protected $createTime = 'createtime';
    protected $updateTime = 'updatetime';

    // 指定软删除字段
    protected $deleteTime = 'deletetime';

    // 追加字段
    protected $append = ['user_like', 'likes_count'];

    // 用户关联
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'id')
            ->field('id,username,headico');
    }

    // 回复关联
    public function replies()
    {
        return $this->hasMany(CourseComment::class, 'parent_id', 'id')
            ->with(['user'])
            ->order('createtime', 'asc');
    }

    // 点赞关联
    public function likes()
    {
        return $this->hasMany(CourseCommentLike::class, 'comment_id', 'id');
    }

    // 获取用户是否点赞
    public function getUserLikeAttr($value, $data)
    {
        $userId = request()->userId ?? 0;
        return $this->likes()->where('user_id', $userId)->count() > 0;
    }

    // 获取点赞数
    public function getLikesCountAttr($value, $data)
    {
        return $this->likes()->count();
    }

    // 获取评论列表
    public static function getCommentList($courseId, $page = 1, $pageSize = 10, $sortBy = 'latest')
    {
        // 基础查询
        $query = self::withCount(['likes'])
            ->with([
                'user' => ['headico'],  // 加载用户信息和头像
                'replies' => function($query) {
                    $query->with(['user' => ['headico']])  // 加载回复的用户信息和头像
                          ->withCount(['likes'])  // 统计回复的点赞数
                          ->append(['user_like', 'likes_count'])
                          ->order('createtime', 'asc');  // 回复按时间正序
                }
            ])
            ->where([
                ['course_id', '=', $courseId],
                ['parent_id', '=', 0]  // 只查询一级评论
            ]);

        // 排序处理
        if ($sortBy === 'likes') {
            $query->order('likes_count', 'desc')
                  ->order('createtime', 'desc');
        } else {
            $query->order('createtime', 'desc');
        }

        // 返回分页数据
        return $query->append(['user_like', 'likes_count'])
            ->paginate([
                'page' => $page,
                'list_rows' => $pageSize
            ]);
    }

    // 获取包含已删除评论的查询（管理员用）
    public static function getListWithTrashed($courseId)
    {
        return self::with(['user'=>['headico'], 'replies'])
            ->where('course_id', $courseId)
            ->where('parent_id', 0)
            ->withTrashed()
            ->select();
    }

    //关联课程
    public function course()
    {
        return $this->belongsTo(ShCourse::class, 'course_id', 'id')
            ->field('id,title');
    }
}