<?php
namespace app\api\middleware;

use think\facade\Db;
use think\facade\Log;

class AddExperience
{
    public function handle($request, \Closure $next)
    {
        $response = $next($request);

        try {
            // 获取当前控制器和方法（格式如：user/login）
            $fullAction = strtolower($request->controller() . '/' . $request->action());
            // 检查是否是登录方法
            $isLoginAction = ($fullAction === 'user/login');

            // 获取经验规则
            $rule = Db::name('user_exp_rule')
                ->where('action', $fullAction)
                ->where('is_open', 1)
                ->where('delete_time', null)
                ->find();

            if (!$rule) {
                return $response;
            }

            // 特殊处理登录方法
            if ($isLoginAction) {
                $this->handleLoginAction($response, $rule);
            } else {
                $this->handleNormalAction($request, $rule);
            }

        } catch (\Exception $e) {
            Log::error('经验系统异常：'.$e->getMessage());
        }

        return $response;
    }

    /**
     * 处理登录方法
     */
    private function handleLoginAction($response, $rule)
    {
        try {
            $responseData = json_decode($response->getContent(), true);

            $token = $responseData['data']['token'] ?? null;
            if (!$token) {
                return;
            }

            $userId = Db::name('user')
                ->where('token', $token)
                ->value('id');
            if ($userId) {
                $this->addExperience($userId, $rule);
            }

        } catch (\Exception $e) {
            Log::error('登录经验处理失败：'.$e->getMessage());
        }
    }

    /**
     * 处理普通方法
     */
    private function handleNormalAction($request, $rule)
    {
        $token = $request->header('token');
        if (!$token) return;

        $userId = Db::name('user')
            ->where('token', $token)
            ->where('delete_time', null)
            ->value('id');

        if ($userId) {
            $this->addExperience($userId, $rule);
        }
    }

    /**
     * 统一添加经验
     */
    private function addExperience($userId, $rule)
    {
        // 检查今日是否已达到上限
        $today = strtotime(date('Y-m-d'));
        $expToday = Db::name('user_exp_log')
            ->where('user_id', $userId)
            ->where('rule_id', $rule['id'])
            ->where('create_time', '>=', $today)
            ->sum('exp');

        if ($rule['daily_limit'] > 0 && $expToday >= $rule['daily_limit']) {
            return;
        }

        // 2. 检查是否在间隔时间内（interval_limit > 0 时才检查）
        if ($rule['interval_limit'] > 0) {
            $lastRecord = Db::name('user_exp_log')
                ->where('user_id', $userId)
                ->where('rule_id', $rule['id'])
                ->order('create_time', 'desc')
                ->find();

            if ($lastRecord && (time() - $lastRecord['create_time']) < $rule['interval_limit']) {
                return; 
            }
        }


        Db::startTrans();
        try {
            // 增加用户经验
            Db::name('user')
                ->where('id', $userId)
                ->inc('experience', $rule['exp'])
                ->update();

            // 记录日志
            Db::name('user_exp_log')->insert([
                'user_id' => $userId,
                'rule_id' => $rule['id'],
                'exp' => $rule['exp'],
                'create_time' => time(),
                'ip' => request()->ip()
            ]);

            Db::commit();
        } catch (\Exception $e) {
            Db::rollback();
            Log::error("经验添加失败：{$userId}-{$rule['id']},msg:{}{$e->getMessage()}");
        }
    }
}