<?php

namespace app\api\controller\manage;

use app\api\middleware\Auth;
use app\api\service\UtilService;
use app\BaseController;
use think\facade\Db;
use think\Request;

class Index extends BaseController
{

    protected $middleware = [
        Auth::class,
    ];

    //看板统计
    public function index(Request $request)
    {
        $userId = $request->userId;

        try {
            Db::startTrans();

            // 1. 获取基础数据
            $courseStats = Db::name('course')
                ->where('user_id', $userId)
                ->field([
                    'COUNT(id) as course_count',
                    'SUM(tvclick) as pv_count',
                    'GROUP_CONCAT(id) as course_ids'
                ])
                ->find();

            $courseIds = $courseStats['course_ids'] ? explode(',', $courseStats['course_ids']) : [];

            // 2. 获取支付相关统计数据
            $paymentStats = Db::name('payment')
                ->where('sf_id', 'in', $courseIds)
                ->where('order_type', 1)
                ->field([
                    'SUM(CASE WHEN pay_status = 1 THEN pay_amount ELSE 0 END) as order_money_sum',
                    'COUNT(id) as order_count'
                ])
                ->find();

            // 3. 获取项目统计数据
            $projectStats = Db::name('project')
                ->where('user_id', $userId)
                ->field([
                    'COUNT(id) as project_count',
                    'SUM(CASE WHEN status IN (0,1,3) THEN 1 ELSE 0 END) as project_pending_count',
                    'SUM(CASE WHEN status = 2 THEN 1 ELSE 0 END) as project_progress_count',
                    'SUM(CASE WHEN status = 4 THEN 1 ELSE 0 END) as project_finished_count'
                ])
                ->find();

            // 组装返回数据
            $result = [
                'orderMoneySum' => (int)$paymentStats['order_money_sum'] ?? 0,
                'orderCount' => (int)$paymentStats['order_count'] ?? 0,
                'courseCount' => (int)$courseStats['course_count'] ?? 0,
                'pvCount' => (int)$courseStats['pv_count'] ?? 0,
                'projectCount' => (int)$projectStats['project_count'] ?? 0,
                'projectPendingCount' => (int)$projectStats['project_pending_count'] ?? 0,
                'projectProgressCount' => (int)$projectStats['project_progress_count'] ?? 0,
                'projectFinishedCount' => (int)$projectStats['project_finished_count'] ?? 0,
                'projectPendingList'=>$this->getProjectPendingList($userId)
            ];

            Db::commit();

            return $this->returnMsg('success', 1,$result);

        } catch (\Exception $e) {
            Db::rollback();
            return $this->returnMsg('系统错误，请稍后再试');
        }
    }

    public function getProjectPendingList($userId)
    {
        // 1. 查询项目列表
        $projects = \app\model\Project::where('user_id', $userId)
            ->where('status', 'in', [0, 1, 3])
            ->field(['id', 'title', 'status', 'tag_ids'])
            ->select();

        // 2. 收集所有标签ID（去重）
        $allTagIds = [];
        foreach ($projects as $project) {
            if ($project->tag_ids) {
                $ids = explode(',', $project->tag_ids);
                $allTagIds = array_merge($allTagIds, $ids);
            }
        }
        $allTagIds = array_unique($allTagIds);

        // 3. 批量查询所有需要的标签
        $tagsMap = [];
        if (!empty($allTagIds)) {
            $tags = \app\model\ProjectTag::where('is_del', 0)
                ->where('id', 'in', $allTagIds)
                ->column('title', 'id');
            $tagsMap = $tags ?: [];
        }

        // 4. 组装结果
        $result = [];
        foreach ($projects as $project) {
            $tagIds = $project->tag_ids ? explode(',', $project->tag_ids) : [];
            $tags = [];

            foreach ($tagIds as $tagId) {
                if (isset($tagsMap[$tagId])) {
                    $tags[] = [
                        'id' => $tagId,
                        'title' => $tagsMap[$tagId]
                    ];
                }
            }

            $result[] = [
                'id' => $project->id,
                'title' => $project->title,
                'status' => $project->status,
                'tags' => $tags
            ];
        }

        return $result;
    }
}