<?php

namespace app\api\service;

use think\facade\Cache;
use think\facade\Config;
class TokenService
{
    // Token前缀
    const TOKEN_PREFIX = 'user_token:';

    // 过期时间(秒)
    const EXPIRE = 72000; // 20小时

    /**
     * 生成用户Token
     * @param int $userId 用户ID
     * @return string Token字符串
     */
    public static function generateToken($userId,$user)
    {
        // 生成随机Token
        $token = self::createToken();

        // 存储Token信息
        $cacheKey = self::TOKEN_PREFIX . $token;
        $tokenData = [
            'user_id' => $userId,
            'create_time' => time(),
            'expire_time' => time() + self::EXPIRE,
            'user'=>$user
        ];

        // 存入缓存
        Cache::set($cacheKey, $tokenData, self::EXPIRE);

        return $token;
    }





    /**
     * 验证Token是否有效
     * @param string $token Token字符串
     * @return array|bool 成功返回Token数据，失败返回false
     */
    public static function verifyToken($token)
    {
        if (empty($token)) {
            return false;
        }

        $cacheKey = self::TOKEN_PREFIX . $token;
        $tokenData = Cache::get($cacheKey);

        if (empty($tokenData)) {
            return false;
        }

        // 检查是否过期
        if ($tokenData['expire_time'] < time()) {
            // 自动清除过期Token
            Cache::delete($cacheKey);
            return false;
        }

        return $tokenData;
    }

    /**
     * 刷新Token过期时间
     * @param string $token Token字符串
     * @return bool 是否刷新成功
     */
    public static function refreshToken($token)
    {
        $cacheKey = self::TOKEN_PREFIX . $token;
        $tokenData = Cache::get($cacheKey);

        if (empty($tokenData)) {
            return false;
        }

        // 更新过期时间
        $tokenData['expire_time'] = time() + self::EXPIRE;
        Cache::set($cacheKey, $tokenData, self::EXPIRE);

        return true;
    }

    /**
     * 删除Token
     * @param string $token Token字符串
     * @return bool
     */
    public static function deleteToken($token)
    {
        $cacheKey = self::TOKEN_PREFIX . $token;
        return Cache::delete($cacheKey);
    }

    /**
     * 生成随机Token
     * @return string
     */
    protected static function createToken()
    {
        return md5(uniqid(mt_rand(), true));
    }
}