<?php

namespace app\api\service;
use think\facade\Db;
use Yansongda\Pay\Pay;
use think\facade\Log;
class PayService
{
    // 支付方式映射
    const PAY_METHOD_ALIPAY = 1;
    const PAY_METHOD_WECHAT = 2;

    // 支付状态
    const PAY_STATUS_UNPAID = 0;
    const PAY_STATUS_SUCCESS = 1;
    const PAY_STATUS_FAILED = 2;

    // 订单类型
    const ORDER_TYPE_COURSE = 1;
    const ORDER_TYPE_CERT = 2;

    /**
     * 创建支付订单
     * @param int $userId 用户ID
     * @param int $orderId 关联订单ID
     * @param float $amount 支付金额
     * @param int $payMethod 支付方式
     * @param int $orderType 订单类型
     * @param array $reqInfo 请求参数
     * @return array
     */
    public static function createPayment($userId, $orderId, $amount, $payMethod, $orderType, $reqInfo = [])
    {
        try {
            // 生成支付订单号
            $orderNo = UtilService::generateCompactOrderNo($userId,'kc');

            // 支付订单数据
            $paymentData = [
                'order_id' => $orderId,
                'pay_method' => $payMethod,
                'order_no' => $orderNo,
                'pay_status' => self::PAY_STATUS_UNPAID,
                'pay_amount' => $amount,
                'order_price' => $amount,
                'order_type' => $orderType,
                'user_id' => $userId,
                'req_info' => json_encode($reqInfo, JSON_UNESCAPED_UNICODE),
                'createtime' => time(),
                'updatetime' => time(),
                'expire_time' => time() + 7200 // 2小时过期
            ];
            // 写入数据库
            $paymentId = Db::name('payment')->insertGetId($paymentData);

            return [
                'status' => true,
                'data' => [
                    'payment_id' => $paymentId,
                    'order_no' => $orderNo
                ]
            ];
        } catch (\Exception $e) {
            Log::error('创建支付订单失败: ' . $e->getMessage());
            return [
                'status' => false,
                'msg' => '创建支付订单失败'
            ];
        }
    }

    /**
     * 生成支付订单号
     * @param int $payMethod 支付方式
     * @return string
     */
    private static function generateOrderNo($payMethod)
    {
        $prefix = $payMethod == self::PAY_METHOD_ALIPAY ? 'ALI' : 'WX';
        return $prefix . date('YmdHis') . mt_rand(1000, 9999);
    }

    /**
     * 处理支付回调
     * @param string $orderNo 支付订单号
     * @param array $notifyData 回调数据
     * @return bool
     */
    public static function handlePaymentNotify($orderNo, $notifyData)
    {

        Db::startTrans();
        try {
            // 查询支付订单
            $payment = Db::name('payment')
                ->where('order_no', $orderNo)
                ->lock(true)
                ->find();

            if (!$payment) {
                throw new \Exception("支付订单不存在: {$orderNo}");
            }

            // 已处理过的订单不再处理
            if ($payment['pay_status'] == self::PAY_STATUS_SUCCESS) {
                Db::commit();
                return true;
            }

            // 验证金额
            $amount = $payment['pay_method'] == self::PAY_METHOD_ALIPAY ?
                $notifyData['amount']['total'] : ($notifyData['amount']['total'] / 100);

            if (bccomp($amount, $payment['pay_amount'], 2) !== 0) {
                throw new \Exception("金额不一致: 订单{$payment['pay_amount']}, 回调{$amount}");
            }

            // 更新支付状态
            $updateData = [
                'pay_status' => self::PAY_STATUS_SUCCESS,
                'pay_time' => strtotime($notifyData['gmt_payment'] ?? date('Y-m-d H:i:s')),
                'pay_info' => json_encode($notifyData, JSON_UNESCAPED_UNICODE),
                'updatetime' => time()
            ];

            $result = Db::name('payment')
                ->where('order_no', $orderNo)
                ->update($updateData);

            if (!$result) {
                throw new \Exception("更新支付状态失败");
            }

            // 根据订单类型处理业务逻辑
            self::handleBusinessAfterPayment($payment);

            Db::commit();
            return true;
        } catch (\Exception $e) {
            Db::rollback();
            Log::error("支付回调处理失败: {$orderNo} - " . $e->getMessage());
            return false;
        }
    }

    /**
     * 支付成功后处理业务逻辑
     * @param array $payment 支付订单数据
     */
    private static function handleBusinessAfterPayment($payment)
    {
        try {
            if ($payment['order_type'] == self::ORDER_TYPE_COURSE) {
                // 处理课程购买逻辑
                // 例如: 更新课程订单状态, 分配课程权限等
                // Db::name('fj_course_order')->where('id', $payment['order_id'])->update(['status' => 1]);
            } elseif ($payment['order_type'] == self::ORDER_TYPE_CERT) {
                // 处理证书购买逻辑
                // Db::name('fj_cert_order')->where('id', $payment['order_id'])->update(['status' => 1]);
            }

            // 可以添加其他业务逻辑，如发送通知等
        } catch (\Exception $e) {
            Log::error("支付后业务处理失败: " . $e->getMessage());
        }
    }

    /**
     * 获取支付配置
     * @param int $payMethod 支付方式
     * @return array
     */
    public static function getPayConfig($payMethod)
    {
        return $payMethod == self::PAY_METHOD_ALIPAY ?
            config('pay.alipay') : config('pay.wechat');
    }
}