from datetime import datetime
from typing import Optional
from dateutil.relativedelta import relativedelta
from fastapi import Depends, APIRouter, Request
from sqlalchemy.orm import Session
from app import get_db
from app.api.export import schemas, crud
from app.api.statement import crud as statement_crud
from libs.result_format import HttpResultResponse, HttpMessage
from libs.token_verify import login_required

router = APIRouter()


@router.get("/list")
def export_list(page: int,
                size: int,
                source: Optional[str] = "",
                start_time: Optional[str] = "",
                end_time: Optional[str] = "",
                token=Depends(login_required), db: Session = Depends(get_db)):
    """导出记录列表"""
    result, total = crud.get_export_list(db, source, start_time, end_time, page, size)
    return HttpResultResponse(total=total, data=result)


@router.post("/update")
def update_export_list(data: schemas.ExportUpdateData, token=Depends(login_required), db: Session = Depends(get_db)):
    """导出记录修改"""
    result = crud.update_export(db, data)
    if result:
        return HttpResultResponse()
    return HttpResultResponse(code=422, msg=HttpMessage.LOSE)


@router.delete("/delete/{id}")
def delete_export_one(id:int, token=Depends(login_required), db: Session = Depends(get_db)):
    """导出记录删除"""
    is_delete = crud.delete_export(db, id)
    if is_delete:
        return HttpResultResponse()
    return HttpResultResponse(code=422, msg=HttpMessage.DELETE_LOSE)


@router.get("/source")
def export_source_query(db: Session = Depends(get_db), token=Depends(login_required)):
    """导出来源查询"""
    result = crud.get_source_data(db)
    return HttpResultResponse(data=result)


@router.get("/month/total")
def month_query_total_export(page: int,
                             size: int,
                             date: str = "",
                             type: str = "",
                             name: str = "",
                             token=Depends(login_required)):
    """月度表计算"""
    if not date:
        month_date = datetime.now().date() - relativedelta(months=1)
        date = month_date.strftime("%Y%m")
    result, num = crud.CalculationMonthlyBill().month_statistics_task('assets_log_' + date, type, name,
                                                                      page, size)
    return HttpResultResponse(total=num, data=result)


@router.get("/month/details")
def month_query_total_export(key: str,
                             is_income: int,
                             page: int,
                             size: int,
                             date: str = "",
                             token=Depends(login_required)):
    """月度计算,详情"""
    if not date:
        return HttpResultResponse(code=500, msg='查询月份不能为空')
    result, num = crud.CalculationMonthlyDetails().statement_income_expenditure(key=key, is_income=is_income, page=page, size=size, date=date)
    return HttpResultResponse(total=num, data=result)


@router.post("/error/data")
def month_query_error_data(param: schemas.MonthDetails, token=Depends(login_required)):
    """月度计算,异常数据"""
    if not param.date:
        return HttpResultResponse(code=500, msg='查询月份不能为空')
    result = crud.CalculationMonthlyDetails().query_error_data('assets_log_' + param.date, param.key)
    return HttpResultResponse(total=len(result), data=result)


@router.post("/month")
def month_to_export(param: schemas.MonthStatistics, request: Request, token=Depends(login_required),
                    db: Session = Depends(get_db)):
    """月度表导出"""
    headers = request.get("headers")
    if not param.date:
        return HttpResultResponse(code=500, msg='查询月份不能为空')
    result = crud.MonthDataDerive().derive_data('assets_log_' + param.date, param.type, param.name)
    field_head = ['业务名称', '财务编码类型', '出账金额', '入账金额', '是否异常', '异常金额']
    return statement_crud.data_to_file(db, result, "业务类型汇总", headers, field_head)


@router.get("/referenceType/total")
def reference_type_total(date: str, type: str):
    """消费类型分类汇总"""
    if not all([date, type]):
        return HttpResultResponse(code=500, msg='缺少必传参数')
    result = crud.ReferenceTypeClassification(date, type).classification_summary()
    return HttpResultResponse(data=result)
