from datetime import datetime
import pandas as pd
from sqlalchemy import and_, func
from sqlalchemy.orm import Session
from app.api.account import schemas
from libs.functions import wrapper_out
from libs.orm import QueryAllData
from models import account as models
from models.account import AccountFinance, AccountFinanceDetails, AccountType


def get_account(db: Session, name: str):
    """查询单个"""
    return db.query(models.Account).filter(models.Account.name == name).first()


def get_id_to_authority(db: Session, role_id: int):
    return db.query(models.Account).filter(models.Account.id == role_id).first()


def get_account_list(db: Session, param):
    """列表"""
    result_list = []

    if param.name:
        count = db.query(func.count(models.Account.id)).filter(models.Account.name.like(f'%{param.name}%')).scalar()
        query_res = db.query(models.Account).filter(models.Account.name.like(f'%{param.name}%')).order_by(models.Account.id.desc()).\
            offset((int(param.page) - 1) * param.size).limit(param.page * param.size).all()
    else:
        count = db.query(func.count(models.Account.id)).scalar()
        query_res = db.query(models.Account).order_by(models.Account.id.desc()).offset((int(param.page) - 1) * param.size).limit(param.page * param.size).all()
    if not query_res:
        return []
    for i in query_res:
        serializer_info = i.to_dict()
        if i.income:
            income_list = serializer_info.get('income').split(',')
            serializer_info['income'] = [int(i) for i in income_list]
        else:
            serializer_info['income'] = []
        if i.output:
            output_list = serializer_info.get('output').split(',')
            serializer_info['output'] = [int(i) for i in output_list]
        else:
            serializer_info['output'] = []
        result_list.append(serializer_info)
    return result_list, count


def get_gift_type(db: Session):
    """礼物类型"""
    out_list = []
    income_list = []
    output = db.query(models.AccountType).all()
    for i in output:
        try:
            zer_info = i.to_dict(only=('id', 'key_name', 'key_value', 'type'))
        except:
            zer_info = i
        if zer_info.get("type") == 1:
            income_list.append(zer_info)
        else:
            out_list.append(zer_info)
    return {"income": income_list, "output": out_list}


def create_account(db: Session, param: schemas.AccountCreate):
    """创建"""
    try:
        db_account = models.Account(name=param.name, unique_tag=param.unique_tag, config_key=param.config_key,
                                    remark=param.remark, income=','.join(map(str, param.income)),
                                    output=','.join(map(str, param.output)), create_time=datetime.now())
        db.add(db_account)
        db.commit()
        db.refresh(db_account)
    except Exception as e:
        print(e)
        return {}
    return db_account


def update_account_info(db: Session, old_data):
    """修改"""
    db.query(models.Account).filter(models.Account.id == old_data.id).update(
        {models.Account.name: old_data.name,
         models.Account.remark: old_data.remark,
         models.Account.income: ','.join(map(str, old_data.income)),
         models.Account.output: ','.join(map(str, old_data.output))})
    db.commit()


def get_finance_info(db, data, is_list=None):
    """账户财务信息"""
    finance_condition = []
    finance_condition.append(AccountFinance.account_id == data.aid)
    if data.start_time:
        finance_condition.append(AccountFinance.create_time >= data.start_time)
    if data.end_time:
        finance_condition.append(AccountFinance.create_time <= data.end_time)
    try:
        get_finance_orm = db.query(AccountFinance).filter(and_(*finance_condition))
        condition_data = db.execute(get_finance_orm).fetchall()
        serializer_info = [i[0].to_dict() for i in condition_data]
        serializer_info.reverse()
    except Exception as e:
        print(e)
        return [], 0 if is_list else []
    # 判断是列表还是导出接口
    if is_list:
        return serializer_info[(int(data.page) - 1) * data.size:data.size * data.page], len(serializer_info)
    else:
        return serializer_info


def get_finance_details(db, data, is_list=None):
    """账户财务明细"""
    finance_condition = []
    if data.type or data.type == 0:
        finance_condition.append(AccountFinanceDetails.type == data.type)
    if data.gift_type:
        finance_condition.append(AccountFinanceDetails.gift_type == data.gift_type)
    if data.start_time:
        finance_condition.append(AccountFinanceDetails.create_time >= data.start_time)
    if data.end_time:
        finance_condition.append(AccountFinanceDetails.create_time <= data.end_time)
    try:
        get_details_orm = db.query(AccountFinanceDetails).filter(and_(*finance_condition))
        condition_data = db.execute(get_details_orm).fetchall()
        serial_info = [i[0].to_dict() for i in condition_data]
        serial_info.reverse()
    except Exception as e:
        print(e)
        return [], 0, 0 if is_list else []
    # 判断是列表还是导出接口
    if is_list:
        if not serial_info:
            return [], 0, 0
        df = pd.DataFrame(serial_info)
        count = df['amount'].apply(lambda x: x).sum()
        return serial_info[(int(data.page) - 1) * data.size:data.size * data.page], len(serial_info), count
    else:
        return serial_info


def get_account_type(db: Session, data):
    """礼物类型配置列表"""
    finance_filters = []
    if data.get("key_name"):
        finance_filters.append(AccountType.key_name == data.get("key_name"))
    if data.get("key_value"):
        finance_filters.append(AccountType.key_value == data.get("key_value"))
    if data.get("type") or data.get("type") == 0:
        finance_filters.append(AccountType.type == data.get("type"))
    if data.get("start_time"):
        finance_filters.append(AccountType.create_time >= data.get("start_time") + " 00:00:00")
    if data.get("end_time"):
        finance_filters.append(AccountType.create_time <= data.get("end_time") + " 23:59:59")
    querydata, count = QueryAllData(db, AccountType, data, finance_filters).query_data()
    data = [QueryAllData.serialization(item) for item in querydata]
    return data, count


def update_account_type(db: Session, data):
    """修改账目类型"""
    try:
        db.query(AccountType).filter(AccountType.id == data.id).update({AccountType.key_name: data.key_name,
                                                                        AccountType.key_value: data.key_value,
                                                                        AccountType.type: data.type})
        db.commit()
        return True
    except Exception as e:
        print(e)
        return False


def get_account_type_value(db: Session, key):
    """新建类型 查询是否有重复数据"""
    return db.query(AccountType).filter(AccountType.key_value == key).first()


def create_type(db: Session, data):
    """创建新账目类型"""
    try:
        db_type = AccountType(key_name=data.key_name, key_value=data.key_value, type=data.type,
                              create_time=datetime.now())
        db.add(db_type)
        db.commit()
        db.refresh(db_type)
    except Exception as e:
        print(e)
        return {}
    return db_type
