from concurrent.futures.thread import ThreadPoolExecutor

from sqlalchemy.orm import Session
from core.config.env import env
from libs.business import TYPE_NAME
from libs.db_link import LinkMysql
from libs.functions import get_now_datetime
from libs.orm import QueryAllData
from models.export import ExportFile
import pandas as pd
from starlette.responses import StreamingResponse


def get_export_list(db: Session, source, start_time, end_time, page, size):
    """导出列表"""
    export_filters = []
    if source:
        export_filters.append(ExportFile.source == source)
    if start_time:
        export_filters.append(ExportFile.create_time >= start_time + " 00:00:00")
    if end_time:
        export_filters.append(ExportFile.create_time < end_time + " 24:00:00")
    param = {"source": source, "start_time": start_time, "end_time": end_time, "page": page, "size": size}
    querydata, count = QueryAllData(db, ExportFile, param, export_filters).query_data()
    data = [QueryAllData.serialization(item) for item in querydata]
    return data, count


def update_export(db: Session, data):

    try:
        db.query(ExportFile).filter(ExportFile.id == data.id).update({ExportFile.remark: data.remark})
        db.commit()
    except Exception as e:
        print(e)
        return False
    return True


def delete_export(db: Session, export_id):
    """删除导出记录"""
    try:
        db.query(ExportFile).filter(ExportFile.id == export_id).delete()
        db.commit()
    except Exception as e:
        print(e)
        return False
    return True


def create_export_data(db: Session, export, operator):
    """创建导出记录"""
    try:
        db_export = ExportFile(operator=operator.get("username"), source=export.get("source"), method=export.get("method"),
                               create_time=get_now_datetime(), update_time=get_now_datetime(), status=export.get("status"))
        db.add(db_export)
        db.commit()
        db.refresh(db_export)
    except Exception as e:
        print(e)
        return {}
    return db_export


def get_source_data(db):
    res = db.query(ExportFile.source).filter().group_by('source').all()
    return [i[0] for i in res]


class CalculationMonthlyBill(object):
    """月度计算，出入账目"""
    def __init__(self):
        self.structure_list = []
        self.structure_key = []

    def month_statistics_task(self, date, key_type, name, page, size):
        """主函数"""
        if key_type and not name:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where reference_type='{key_type}' GROUP BY reference_type, type ORDER BY reference_type"
        if name and not key_type:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where reference_type='{name}' GROUP BY reference_type, type ORDER BY reference_type"
            for k, v in TYPE_NAME.items():
                if v == name:
                    sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where reference_type='{k}' GROUP BY reference_type, type ORDER BY reference_type"
                    break
        if name and key_type:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where reference_type='{key_type}' GROUP BY reference_type, type ORDER BY reference_type"
        if not name and not key_type:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} GROUP BY reference_type, type ORDER BY reference_type"
        try:
            res_data = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        except Exception as e:
            return [], 0
        for res in res_data:
            if res["reference_type"] in self.structure_key:
                continue
            if res["reference_type"] in TYPE_NAME:
                name = TYPE_NAME[res["reference_type"]]
            else:
                name = res["reference_type"]
            out = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 0]
            income = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 1]
            out_value = float(out[0]) if out else 0
            income_value = float(income[0]) if income else 0
            a = {
                "name": name,
                "type": res["reference_type"],
                "expenditure": out_value,
                "income": income_value,
                "is_error": 0 if out_value == income_value else 1,
                "error_money": float('%.2f' % (out_value - income_value))
            }
            self.structure_key.append(res["reference_type"])
            self.structure_list.append(a)
        # return self.data_to_table(self.structure_list) # 导出接口
        return self.structure_list[(page-1)*size:page*size], len(self.structure_list)


class CalculationMonthlyDetails(object):
    """月度详情"""
    @staticmethod
    def data_query(db, date, reference_type, is_out, page, size):
        sql = f"SELECT reference_type,order_number,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where reference_type='{reference_type}' and type={is_out} LIMIT {(page-1)*size},{size}"
        res_data = LinkMysql(db).query_mysql(sql)
        return res_data

    @staticmethod
    def acquired_total(db, date, reference_type, is_out):
        sql = f"SELECT reference_type FROM {date} where reference_type='{reference_type}' and type={is_out}"
        result = LinkMysql(db).query_mysql(sql)
        return len(result)

    def statement_income_expenditure(self, **param):

        with ThreadPoolExecutor(max_workers=2) as pool:
            future1 = pool.submit(self.data_query, env.DB_HISTORY, 'assets_log_' + param.get("date"), param.get("key"), param.get("is_income"), param.get("page"), param.get("size"))
            future2 = pool.submit(self.acquired_total, env.DB_HISTORY, 'assets_log_' + param.get("date"), param.get("key"), param.get("is_income"))
        data = future1.result()
        num = future2.result()
        return data, num

    @staticmethod
    def query_error_data(date, reference_type):

        group_sql = f"SELECT order_id, COUNT(order_id) as num FROM {date} where reference_type='{reference_type}' GROUP BY order_id"
        result = LinkMysql(env.DB_HISTORY).query_mysql(group_sql)
        error_list = [str(i.get("order_id")) for i in result if i.get("num") != 2]
        if len(error_list) == 1:
            sql = f"SELECT uuid,reference_type,order_number,order_id,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where order_id={error_list[0]}"
        if len(error_list) > 1:
            sql = f"SELECT uuid,reference_type,order_number,order_id,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where order_id in({','.join(error_list)})"
        if len(error_list) == 0:
            return []
        result = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        return result


class MonthDataDerive(object):
    """月度导出"""
    def __init__(self):
        self.derive_key = []
        self.derive_list = []

    @staticmethod
    def data_to_table(data):
        """数据导出"""
        bk = pd.DataFrame(data)
        with pd.ExcelWriter(f'static/业务类型月度汇总报表.xlsx') as writer:
            bk.to_excel(writer, sheet_name='业务类型月度汇总', index=False)
        file = open(writer, 'rb')

        return StreamingResponse(file, media_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet')

    def derive_data(self, date, key_type):
        if key_type:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where reference_type='{key_type}' GROUP BY reference_type, type ORDER BY reference_type"
        else:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} GROUP BY reference_type, type ORDER BY reference_type"
        try:
            res_data = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        except Exception as e:
            return [], 0
        for res in res_data:
            if res["reference_type"] in self.derive_key:
                continue
            if res["reference_type"] in TYPE_NAME:
                name = TYPE_NAME[res["reference_type"]]
            else:
                name = res["reference_type"]
            out = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 0]
            income = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 1]
            out_value = float(out[0]) if out else 0
            income_value = float(income[0]) if income else 0
            a = {
                "name": name,
                "type": res["reference_type"],
                "expenditure": out_value,
                "income": income_value,
                "is_error": '否' if out_value == income_value else '是',
                "error_money": float('%.2f' % (out_value - income_value))
            }
            self.derive_key.append(res["reference_type"])
            self.derive_list.append(a)

        return self.derive_list
