from sqlalchemy import Column, Integer, String, Text, DateTime, Float
from core.storage.db import Base
from sqlalchemy_serializer import SerializerMixin

from libs import functions


class Account(Base, SerializerMixin):
    """账户表"""
    __tablename__ = "account"

    id = Column(Integer, primary_key=True, index=True)
    name = Column(String(255), index=True, comment="账户名称")
    unique_tag = Column(String(100), unique=True, comment="唯一标记")
    beneficiary = Column(String(100), comment="受益人")
    consumable = Column(String(255), comment="账户余额")
    remark = Column(String(255), comment="描述")
    uuid = Column(String(60), default=functions.uuid(), comment="uuid")
    config_key = Column(String(150), comment="配置key,和api接口约定好的标识")
    income = Column(Text, comment="入账")
    output = Column(Text, comment="出账")
    operator = Column(String(255), comment="操作员（每笔转账都要记录一下）")
    operator_id = Column(Integer, comment="操作员id（可能设计到改了名字，记id比较好）")
    create_time = Column(DateTime, comment="创建时间")

    __table_args__ = {'comment': '财务系统-账号管理'}


class AccountType(Base, SerializerMixin):
    """礼物类型配置"""
    __tablename__ = "account_type"

    id = Column(Integer, primary_key=True, index=True)
    key_name = Column(String(255), comment="礼物类型名字")
    key_value = Column(String(255), comment="礼物类型value")
    type = Column(Integer, comment="方式：0出账，1：入账")
    create_time = Column(DateTime, comment="创建时间")

    __table_args__ = {'comment': '礼物类型配置'}


class AccountFinance(Base, SerializerMixin):
    """账号财务数据(单位：天)"""
    __tablename__ = "account_finance"

    id = Column(Integer, primary_key=True, index=True)
    account_id = Column(Integer, comment="账户id")
    account_tag = Column(String(255), comment="账户标识")
    account_uuid = Column(String(100), comment="账户uuid")
    balance = Column(String(255), comment="当日余额")
    income = Column(Float, comment="当日入账")
    output = Column(Float, comment="当日出账")
    create_time = Column(DateTime, comment="创建时间")

    __table_args__ = {'comment': '账号财务数据表'}


class AccountFinanceDetails(Base, SerializerMixin):
    """账号财务数据详情"""
    __tablename__ = "account_finance_details"

    id = Column(Integer, primary_key=True, index=True)
    account_id = Column(Integer, comment="账户id")
    account_tag = Column(String(255), comment="账户标识")
    account_uuid = Column(String(100), comment="账户uuid")
    order_number = Column(String(255), comment="订单号")
    gift_type = Column(String(255), comment="礼物类型")
    type = Column(Integer, comment="方式：0出账，1：入账")
    amount = Column(Float, comment="金额")
    create_time = Column(DateTime, comment="创建时间")

    __table_args__ = {'comment': '账号财务数据详情表'}
