from concurrent.futures.thread import ThreadPoolExecutor

from sqlalchemy.orm import Session
from core.config.env import env
from libs.business import TYPE_NAME, query_fi_account_type
from libs.db_link import LinkMysql
from libs.functions import get_now_datetime
from libs.orm import QueryAllData
from models.export import ExportFile
import pandas as pd
from xpinyin import Pinyin
from starlette.responses import StreamingResponse


def get_export_list(db: Session, source, start_time, end_time, page, size):
    """导出列表"""
    export_filters = []
    if source:
        export_filters.append(ExportFile.source == source)
    if start_time:
        export_filters.append(ExportFile.create_time >= start_time + " 00:00:00")
    if end_time:
        export_filters.append(ExportFile.create_time < end_time + " 24:00:00")
    param = {"source": source, "start_time": start_time, "end_time": end_time, "page": page, "size": size}
    querydata, count = QueryAllData(db, ExportFile, param, export_filters).query_data()
    data = [QueryAllData.serialization(item) for item in querydata]
    return data, count


def update_export(db: Session, data):
    try:
        db.query(ExportFile).filter(ExportFile.id == data.id).update({ExportFile.remark: data.remark})
        db.commit()
    except Exception as e:
        print(e)
        return False
    return True


def delete_export(db: Session, export_id):
    """删除导出记录"""
    try:
        db.query(ExportFile).filter(ExportFile.id == export_id).delete()
        db.commit()
    except Exception as e:
        print(e)
        return False
    return True


def create_export_data(db: Session, export, operator):
    """创建导出记录"""
    try:
        db_export = ExportFile(operator=operator.get("username"), source=export.get("source"), method=export.get("method"),
                               create_time=get_now_datetime(), update_time=get_now_datetime(), status=export.get("status"))
        db.add(db_export)
        db.commit()
        db.refresh(db_export)
    except Exception as e:
        print(e)
        return {}
    return db_export


def get_source_data(db):
    res = db.query(ExportFile.source).filter().group_by('source').all()
    return [i[0] for i in res]


class CalculationMonthlyBill(object):
    """月度计算，出入账目"""
    def __init__(self):
        self.structure_list = []
        self.structure_key = []

    def month_statistics_task(self, date, key_type, name, page, size):
        """主函数"""
        assert_list = []
        if name:
            k_list = []
            type_name = query_fi_account_type()
            for k, v in type_name.items():
                if v == name or name in v:
                    k_list.append(k)
            if len(k_list) > 1:
                assert_list.append(f" reference_type in{tuple(k_list)}")
            if len(k_list) == 1:
                assert_list.append(f" reference_type='{k_list[0]}'")
            if not k_list:
                py = Pinyin().get_pinyin(name)
                assert_list.append(f" reference_type='{py.replace('-', '')}'")
        if key_type:
            assert_list.append(f" reference_type like '%{key_type}%'")
        if assert_list:
            sql = f"SELECT reference_type, type, SUM(amount)/1000 as money FROM {date} where {' and '.join(assert_list)} GROUP BY reference_type, type ORDER BY reference_type"
        else:
            sql = f"SELECT reference_type, type, SUM(amount)/1000 as money FROM {date} GROUP BY reference_type, type ORDER BY reference_type"
        try:
            res_data = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        except Exception as e:
            return [], 0
        type_name = query_fi_account_type()
        for res in res_data:
            if res["reference_type"] in self.structure_key:
                continue
            if res["reference_type"] in type_name:
                name = type_name[res["reference_type"]]
            else:
                name = res["reference_type"]
            out = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 0]
            income = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 1]
            out_value = out[0] if out else 0
            income_value = income[0] if income else 0
            a = {
                "name": name,
                "type": res["reference_type"],
                "expenditure": out_value,
                "income": income_value,
                "is_error": 0 if out_value == income_value else 1,
                "error_money": float('%.3f' % (out_value - income_value))
            }
            self.structure_key.append(res["reference_type"])
            self.structure_list.append(a)
        # return self.data_to_table(self.structure_list) # 导出接口
        return self.structure_list[(page-1)*size:page*size], len(self.structure_list)


class CalculationMonthlyDetails(object):
    """月度详情"""
    @staticmethod
    def data_query(db, date, reference_type, is_out, page, size):
        sql = f"SELECT uuid,reference_type,reference_number,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where reference_type='{reference_type}' and type={is_out} LIMIT {(page-1)*size},{size}"
        res_data = LinkMysql(db).query_mysql(sql)
        return res_data

    @staticmethod
    def acquired_total(db, date, reference_type, is_out):
        sql = f"SELECT reference_type FROM {date} where reference_type='{reference_type}' and type={is_out}"
        result = LinkMysql(db).query_mysql(sql)
        return len(result)

    @staticmethod
    def calculation_total(db, date, reference_type, is_out):
        sql = f"SELECT sum(amount)/1000 as amount FROM {date} where reference_type='{reference_type}' and type={is_out}"
        result = LinkMysql(db).query_mysql(sql)
        return result

    def statement_income_expenditure(self, **param):

        with ThreadPoolExecutor(max_workers=3) as pool:
            future1 = pool.submit(self.data_query, env.DB_HISTORY, 'assets_log_' + param.get("date"), param.get("key"), param.get("is_income"), param.get("page"), param.get("size"))
            future2 = pool.submit(self.acquired_total, env.DB_HISTORY, 'assets_log_' + param.get("date"), param.get("key"), param.get("is_income"))
            future3 = pool.submit(self.calculation_total, env.DB_HISTORY, 'assets_log_' + param.get("date"), param.get("key"), param.get("is_income"))
        data = future1.result()
        num = future2.result()
        total = future3.result()
        if data:
            return data, num, float(total[0]['amount'])
        return [], 0, 0

    @staticmethod
    def query_error_data(date, reference_type):

        group_sql = f"SELECT order_id, COUNT(order_id) as num FROM {date} where reference_type='{reference_type}' GROUP BY order_id"
        result = LinkMysql(env.DB_HISTORY).query_mysql(group_sql)
        error_list = [str(i.get("order_id")) for i in result if i.get("num") != 2]
        if len(error_list) == 1:
            sql = f"SELECT uuid,reference_type,order_number,order_id,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where order_id={error_list[0]}"
        if len(error_list) > 1:
            sql = f"SELECT uuid,reference_type,order_number,order_id,type,cast(amount as decimal(20,6))/1000 as money,amount_type,create_time FROM {date} where order_id in({','.join(error_list)})"
        if len(error_list) == 0:
            return []
        result = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        return result


class MonthDataDerive(object):
    """月度导出"""
    def __init__(self):
        self.derive_key = []
        self.derive_list = []

    @staticmethod
    def data_to_table(data):
        """数据导出"""
        bk = pd.DataFrame(data)
        with pd.ExcelWriter(f'static/业务类型月度汇总报表.xlsx') as writer:
            bk.to_excel(writer, sheet_name='业务类型月度汇总', index=False)
        file = open(writer, 'rb')

        return StreamingResponse(file, media_type='application/vnd.openxmlformats-officedocument.spreadsheetml.sheet')

    def derive_data(self, date, key_type, name):
        assert_list = []
        if name:
            k_list = []
            type_name = query_fi_account_type()
            for k, v in type_name.items():
                if v == name or name in v:
                    k_list.append(k)
            if len(k_list) > 1:
                assert_list.append(f" reference_type in{tuple(k_list)}")
            if len(k_list) == 1:
                assert_list.append(f" reference_type='{k_list[0]}'")
        if key_type:
            assert_list.append(f" reference_type like '%{key_type}%'")
        if assert_list:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} where {' and '.join(assert_list)} GROUP BY reference_type, type ORDER BY reference_type"
        else:
            sql = f"SELECT reference_type, type, SUM(cast(amount as decimal(20,6)))/1000 as money FROM {date} GROUP BY reference_type, type ORDER BY reference_type"
        try:
            res_data = LinkMysql(env.DB_HISTORY).query_mysql(sql)
        except Exception as e:
            return []
        for res in res_data:
            if res["reference_type"] in self.derive_key:
                continue
            type_name = query_fi_account_type()
            if res["reference_type"] in type_name:
                name = type_name[res["reference_type"]]
            else:
                name = res["reference_type"]
            out = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 0]
            income = [i['money'] for i in res_data if i['reference_type'] == res["reference_type"] and i['type'] == 1]
            out_value = float(out[0]) if out else 0
            income_value = float(income[0]) if income else 0
            a = {
                "name": name,
                "type": res["reference_type"],
                "expenditure": out_value,
                "income": income_value,
                "is_error": '否' if out_value == income_value else '是',
                "error_money": float('%.2f' % (out_value - income_value))
            }
            self.derive_key.append(res["reference_type"])
            self.derive_list.append(a)

        return self.derive_list


class ReferenceTypeClassification():
    """消费类型再分类汇总"""
    def __init__(self, date, reference_type):
        self.date = 'assets_log_' + date
        self.reference_type = reference_type
        self.outcome = []
        self.income = []

    def classification_summary(self):

        data_sql = f"select uuid,type,sum(amount) as amount,create_time,reference_type,amount_type from {self.date} where reference_type='{self.reference_type}' GROUP BY uuid,type"
        guild_sql = f"select uuid from guild"
        account_sql = f"select uuid,name from fi_account"
        anchor_sql = f"select uuid from v2_user where is_achor in(1,2)"
        with ThreadPoolExecutor(max_workers=4) as pool:
            future1 = pool.submit(LinkMysql(env.DB_HISTORY).query_mysql, data_sql)
            future2 = pool.submit(LinkMysql(env.DB_3YV2).query_mysql, guild_sql)
            future3 = pool.submit(LinkMysql(env.DB_3YV2).query_mysql, account_sql)
            future4 = pool.submit(LinkMysql(env.DB_3YV2).query_mysql, anchor_sql)
        data = future1.result()
        guild = future2.result()
        account = future3.result()
        anchor = future4.result()
        # 公会uuid列表
        guild_data = [i.get('uuid') for i in guild]
        # 主播uuid列表
        anchor_data = [i.get('uuid') for i in anchor]
        # 系统账户格式化{'系统账户uuid':'系统账户名字'}
        account_dict = {}
        for k in account:
            account_dict[k.get('uuid')] = k.get('name')
        # 数据分类
        for op in data:
            op['money'] = round(float(op['amount']), 2)
            if op.get('uuid') in guild_data:
                op['nickname'] = '公会'
            elif op.get('uuid') in anchor_data:
                if op.get('reference_type') == 'studioGift':
                    if op.get('amount_type') == 'backpack':
                        op['nickname'] = '背包送出'
                    elif op.get('amount_type') == 'consumable':
                        op['nickname'] = '钻石送出'
                    else:
                        op['nickname'] = '主播'
                else:
                    op['nickname'] = '主播'
            elif account_dict.get(op['uuid']):
                op['nickname'] = account_dict.get(op['uuid'])
            else:
                if op.get('reference_type') == 'studioGift':
                    if op.get('amount_type') == 'backpack':
                        op['nickname'] = '背包送出'
                    elif op.get('amount_type') == 'consumable':
                        op['nickname'] = '钻石送出'
                    else:
                        op['nickname'] = '用户'
                else:
                    op['nickname'] = '用户'
            if op.get('type') == 0:
                self.outcome.append(op)
            else:
                self.income.append(op)
        print(f"出账：{self.outcome}, 入账：{self.income}")
        res_list = []
        if self.outcome:
            outcome = pd.DataFrame(self.outcome)
            ogs = outcome.groupby("nickname").sum()
            for k, v in ogs.to_dict().get('money').items():
                res_list.append({"type": "出账", "name": k, "money": round(float(v/1000), 3)})
        if self.income:
            income = pd.DataFrame(self.income)
            igs = income.groupby("nickname").sum()
            for k, v in igs.to_dict().get('money').items():
                res_list.append({"type": "入账", "name": k, "money": round(float(v/1000), 3)})
        return res_list


class AbnormalDataDetails(object):
    """业务类型汇总异常数据详情"""
    def __init__(self, date, reference_type):
        self.date = 'assets_log_' + date
        self.reference_type = reference_type

    def abnormal_task(self):
        out_sql = f"select order_number from {self.date} where reference_type='{self.reference_type}' and type=0"
        income_sql = f"select order_number from {self.date} where reference_type='{self.reference_type}' and type=1"
        with ThreadPoolExecutor(max_workers=2) as pool:
            out_res = pool.submit(LinkMysql(env.DB_HISTORY).query_mysql, out_sql)
            income_res = pool.submit(LinkMysql(env.DB_HISTORY).query_mysql, income_sql)
        out_data = out_res.result()
        income_data = income_res.result()
        out_order_number = [i['order_number'] for i in out_data]
        income_order_number = [i['order_number'] for i in income_data]
        error_order_number = list(set(out_order_number) ^ set(income_order_number))
        if len(error_order_number) == 1:
            error_sql = f"select uuid,type,reference_type,reference_number,order_number,amount/1000 as amount,create_time from {self.date} where order_number='{error_order_number[0]}'"
        elif len(error_order_number) > 1:
            error_sql = f"select uuid,type,reference_type,reference_number,order_number,amount/1000 as amount,create_time from {self.date} where order_number in{tuple(error_order_number)}"
        else:
            return []
        res = LinkMysql(env.DB_HISTORY).query_mysql(error_sql)
        return res
