import json
from datetime import datetime
from typing import Optional

from app.api.statement import crud, schemas, guild
from app.api.account import schemas as acc_schemas
from app import get_db
from fastapi import Depends, APIRouter, File, Request
from sqlalchemy.orm import Session
from app.api.statement.crud import RechargeStatement, WithdrawStatement
from app.api.statement.guild import GuildSet, paymentset_guild_data, outon_account_data, accout_list_data, \
    query_uuid_or_user_number, account_money, transfer_money, transfer_query
from app.api.statement.schemas import PaymentWithdrawalList, PaymentAdd, PaymentAccountlList, UserNumber, CreateBill
from libs.img_code import new_upload_file, random_number
from libs.result_format import HttpResultResponse
from libs.token_verify import login_required
from models.recharge import Fitransferlog
from models.account import Account

router = APIRouter()


@router.get("/recharge/list")
def statement_recharge_list(page: Optional[int] = 0, size: Optional[int] = 10, order_number: Optional[str] = "",
                            uuid: Optional[int] = "", sid: Optional[str] = "", start_time: Optional[str] = '',
                            end_time: Optional[str] = "", types: Optional[str] = "", reference_type: Optional[str] = "",
                            month_type: Optional[int] = "", token=Depends(login_required)):
    """充值报表列表"""
    total, statement_list, money = RechargeStatement().query_data(page, size, order_number, uuid, sid, start_time,
                                                                  end_time, types, reference_type, month_type)
    return HttpResultResponse(total=total, count=float(money), data=statement_list)


@router.post("/derive/excel")
def statement_derive_excel(data: schemas.StatementList, request: Request, db: Session = Depends(get_db),
                           token=Depends(login_required)):
    """充值报表导出  需要修改"""
    header_list = request.get("headers")
    statement_list = RechargeStatement().get_statements(data)
    return crud.data_to_file(db, statement_list, "充值报表", header_list)


@router.get("/userWithdrawal/list")
def user_withdrawal_list(page: Optional[int] = 0, size: Optional[int] = 10, uuid: Optional[int] = '',
                         status: Optional[int] = '', start_time: Optional[str] = '', end_time: Optional[str] = "",
                         month_type: Optional[int] = "", token=Depends(login_required)):
    """用户提现列表"""
    total, statement_list, money = WithdrawStatement().get_user_withdraw_cash(page, size, uuid, status, start_time,
                                                                              end_time, month_type)
    return HttpResultResponse(total=total, count=float(money), data=statement_list)


@router.get("/guildWithdrawal/list")
def guild_withdrawal_list(page: Optional[int] = 0, size: Optional[int] = 10, guild_id: Optional[int] = '',
                          status: Optional[int] = '', start_time: Optional[str] = '', end_time: Optional[str] = "",
                          month_type: Optional[int] = "", token=Depends(login_required)):
    """公会提现列表  暂无"""
    total, statement_list, money = WithdrawStatement().get_guild_withdraw_cash(page, size, guild_id, status, start_time,
                                                                               end_time, month_type)
    return HttpResultResponse(total=total, count=float(money), data=statement_list)


@router.get("/guild/settlement")
def guild_settlement_list(page: Optional[int] = 0, size: Optional[int] = 10, status: Optional[int] = '',
                          start_time: Optional[str] = '', end_time: Optional[str] = "", guild_id: Optional[str] = "",
                          token=Depends(login_required)):
    """公会结算"""
    guild_list, total = GuildSet().get_guild_data(page, size, status, start_time, end_time, guild_id)
    return HttpResultResponse(total=total, data=guild_list)


@router.get("/guild/payment")
def guild_payment_list(page: Optional[int] = 1, size: Optional[int] = 10, start_time: Optional[str] = '',
                       end_time: Optional[str] = "", token=Depends(login_required)):
    """打款记录"""
    payment_list, total = paymentset_guild_data(page, size, start_time, end_time)
    return HttpResultResponse(total=total, data=payment_list)


@router.post('/upload')
async def uploadFile(a_file: bytes = File(...), token=Depends(login_required), ):
    a_content = a_file
    image_name = random_number()
    cos_path = new_upload_file(a_content, image_name)
    if not cos_path:
        return HttpResultResponse(code=500, msg="上传失败,请检查cos是否正常")
    return HttpResultResponse(data=cos_path)


@router.post("/payment/add")
def guild_payment_list(data: PaymentAdd, db: Session = Depends(get_db)):
    """提交打款"""
    transfer_query(data)
    return HttpResultResponse(code=200, data='')


@router.get("/onaccount")
def outon_account(db: Session = Depends(get_db), token=Depends(login_required)):
    """转出和转入账户人员列表"""
    account_list, total = outon_account_data(db, Account, {})
    return HttpResultResponse(total=total, data=account_list)


@router.get("/transferlist")
def out_on_account(page: int,
                   size: int,
                   start_time: Optional[str] = "",
                   end_time: Optional[str] = "",
                   transfer_name: Optional[str] = "",
                   dst_name: Optional[str] = "",
                   token=Depends(login_required)):
    """转账列表"""
    account_list, total = accout_list_data(page=page, size=size, start_time=start_time, end_time=end_time,
                                           transfer_name=transfer_name, dst_name=dst_name)
    return HttpResultResponse(total=total, data=account_list)


@router.get("/query/consumable")
def query_account_money(uuid: str, amount_type: Optional[str] = "consumable", token=Depends(login_required)):
    """查询账户余额"""
    if not uuid:
        return HttpResultResponse(code=500, msg='请输入uuid')
    consumable = account_money(uuid, amount_type)
    return HttpResultResponse(data=consumable)


@router.post("/query/uuid")
def query_uuid_account(data: UserNumber, token=Depends(login_required)):
    """uuid user_number查询"""
    user_info = query_uuid_or_user_number(data)
    if not user_info:
        return HttpResultResponse(code=500, msg='没有查到数据')
    return HttpResultResponse(data=user_info)


@router.post("/transfer/money")
def account_transfer_money(data: CreateBill, request: Request,
                           token=Depends(login_required), db: Session = Depends(get_db)):
    """账户转账"""
    if not all([data.dst_id, data.transfer_id, data.amount, data.remark]):
        return HttpResultResponse(code=500, msg="缺少必传参数")
    header_list = request.get("headers")
    res_info = transfer_money(db, data, header_list)
    if res_info:
        return HttpResultResponse(code=500, msg=res_info)
    return HttpResultResponse()


@router.post("/guild/update")
def guild_update_list(data: schemas.GuildUpdate, token=Depends(login_required)):
    """公会结算"""
    GuildSet().update_guild_info(data)
    return HttpResultResponse()


@router.get("/finance/fix")
def finance_fix(page: int,
                size: int,
                start_time: Optional[str] = "",
                end_time: Optional[str] = "",
                token=Depends(login_required)):
    """财务修复报表列表"""
    res, num = crud.FinanceFix.get_finance_fix_data(page, size, start_time, end_time)
    return HttpResultResponse(total=num, data=res)


@router.post("/submit/fixTable")
def finance_fix(data: acc_schemas.FixTable, request: Request,
                token=Depends(login_required), db: Session = Depends(get_db)):
    """新增修复报表"""
    if not all([data.money, data.uuid, data.amount_type, data.remark]):
        return HttpResultResponse(code=500, msg='缺少必传参数')
    header_list = request.get("headers")
    res = guild.create_fix_table(db, data, header_list)
    if res:
        return HttpResultResponse(code=500, msg=res)
    return HttpResultResponse()


@router.get("/data/recovery")
def recovery_fix(page: int,
                 size: int,
                 uuid: Optional[str] = '',
                 start_time: Optional[str] = "",
                 end_time: Optional[str] = "",
                 token=Depends(login_required)):
    """异常数据修复列表"""
    recovery_list, total = guild.recovery_fix_data(page=page, size=size, uuid=uuid, start_time=start_time,
                                                   end_time=end_time)
    return HttpResultResponse(total=total, data=recovery_list)


@router.post("/submit/recovery")
def finance_fix(data: acc_schemas.RecoveryTable, token=Depends(login_required), db: Session = Depends(get_db)):
    """新增异常数据修复"""
    if not all([data.uuid, data.out_money, data.entry_money, data.from_time, data.cont]):
        return HttpResultResponse(code=500, msg='缺少必传参数')
    res = guild.create_recovery_table(data)
    if res:
        return HttpResultResponse(code=500, msg=res)
    return HttpResultResponse()


@router.post("/recovery/fix")
def recovery_fix(data: acc_schemas.RecoveryupdateTable, token=Depends(login_required)):
    """修改异常数据"""
    res = guild.update_recovery_table(data)
    if res:
        return HttpResultResponse(code=500, msg=res)
    return HttpResultResponse()
