import json
from datetime import datetime
from app.api.statement import crud, schemas, guild
from app.api.account import schemas as acc_schemas
from app import get_db
from fastapi import Depends, APIRouter, File, Request
from sqlalchemy.orm import Session
from app.api.statement.crud import RechargeStatement, WithdrawStatement
from app.api.statement.guild import GuildSet, paymentset_guild_data, outon_account_data, accout_list_data, \
    query_uuid_or_user_number, account_money, transfer_money
from app.api.statement.schemas import PaymentWithdrawalList, PaymentAdd, PaymentAccountlList, UserNumber, CreateBill
from libs.img_code import new_upload_file, random_number
from libs.result_format import HttpResultResponse
from libs.token_verify import login_required
from models.recharge import Paymentlog, Fitransferlog, Account_log
from models.account import Account

router = APIRouter()


@router.post("/recharge/list")
def statement_recharge_list(data: schemas.StatementList, token=Depends(login_required), db: Session = Depends(get_db)):
    """充值报表列表"""
    statement_list, total, money = RechargeStatement().get_statements(db, data, 1)
    return HttpResultResponse(total=total, count=float(money), data=statement_list)


@router.post("/derive/excel")
def statement_derive_excel(data: schemas.StatementList, request: Request,
                           token=Depends(login_required), db: Session = Depends(get_db)):
    """充值报表导出"""
    header_list = request.get("headers")
    statement_list = RechargeStatement().get_statements(db, data)
    return crud.data_to_file(db, statement_list, "充值报表", header_list)


@router.post("/userWithdrawal/list")
def user_withdrawal_list(data: schemas.UserWithdrawalList, token=Depends(login_required), db: Session = Depends(get_db)):
    """用户提现列表"""
    statement_list, total, money, final_money = WithdrawStatement().get_user_withdraw_cash(db, data)
    return HttpResultResponse(total=total, count=float(money), actual_count=final_money, data=statement_list)


@router.post("/guildWithdrawal/list")
def guild_withdrawal_list(data: schemas.GuildWithdrawalList, token=Depends(login_required), db: Session = Depends(get_db)):
    """公会提现列表"""
    guild_list, total, money, final_money = WithdrawStatement().get_guild_withdraw_cash(db, data)
    return HttpResultResponse(total=total, count=float(money), actual_count=final_money, data=guild_list)


@router.post("/guild/settlement")
def guild_settlement_list(data: schemas.GuildWithdrawalList, token=Depends(login_required), db: Session = Depends(get_db)):
    """公会结算"""
    guild_list, total = GuildSet().get_guild_data(db, data)
    return HttpResultResponse(total=total, data=guild_list)


@router.post("/guild/payment")
def guild_payment_list(data:PaymentWithdrawalList, token=Depends(login_required), db: Session = Depends(get_db)):
    """打款记录"""
    payment_list, total = paymentset_guild_data(db,Paymentlog,data.dict(exclude_none=True))
    return HttpResultResponse(total=total, data=payment_list)


@router.post('/upload')
async def uploadFile(a_file: bytes = File(...)):
    a_content = a_file
    image_name = random_number()
    cos_path = new_upload_file(a_content, image_name)
    if not cos_path:
        return HttpResultResponse(code=500, msg="上传失败,请检查cos是否正常")
    return HttpResultResponse(data=cos_path)


@router.post("/payment/add")
def guild_payment_list(data: PaymentAdd, token=Depends(login_required), db: Session = Depends(get_db)):
    """提交打款"""
    dataPayment = Paymentlog(uuid=data.uuid, money=data.money, cont=data.cont,url=data.url,create_time=datetime.now())
    db.add(dataPayment)
    db.commit()
    db.refresh(dataPayment)
    db.close()
    return HttpResultResponse(data='')


@router.get("/onaccount")
def outon_account(db: Session = Depends(get_db), token=Depends(login_required)):
    """转出和转入账户人员列表"""
    account_list, total = outon_account_data(db, Account, {})
    return HttpResultResponse(total=total, data=account_list)


@router.post("/transferlist")
def outon_account(data: PaymentAccountlList, token=Depends(login_required), db: Session = Depends(get_db)):
    """转账列表"""
    account_list, total = accout_list_data(db, Fitransferlog, data.dict(exclude_none=True))
    return HttpResultResponse(total=total, data=account_list)


@router.post("/query/consumable")
def query_account_money(data: UserNumber, token=Depends(login_required), db: Session = Depends(get_db)):
    """查询账户余额"""
    consumable = account_money(db, data)
    return HttpResultResponse(data=int(consumable))


@router.post("/query/uuid")
def query_uuid_account(data: UserNumber, token=Depends(login_required), db: Session = Depends(get_db)):
    """uuid user_number查询"""
    user_info = query_uuid_or_user_number(db, data)
    return HttpResultResponse(data=user_info)


@router.post("/transfer/money")
def account_transfer_money(data: CreateBill, request: Request,
                           token=Depends(login_required), db: Session = Depends(get_db)):
    """账户转账"""
    if not all([data.dst_id, data.transfer_id, data.amount, data.remark]):
        return HttpResultResponse(code=500, msg="缺少必传参数")
    header_list = request.get("headers")
    res_info = transfer_money(db, data, header_list)
    if res_info:
        return HttpResultResponse(code=500, msg=res_info)
    return HttpResultResponse()


@router.post("/guild/update")
def guild_update_list(data: schemas.GuildUpdate, token=Depends(login_required), db: Session = Depends(get_db)):
    """公会结算"""
    GuildSet().update_guild_info(db, data)
    return HttpResultResponse()


@router.post("/finance/fix")
def finance_fix(data: acc_schemas.PublicModel, token=Depends(login_required), db: Session = Depends(get_db)):
    """账户修复报表"""
    res, num = crud.FinanceFix.get_finance_fix_data(db, data.dict(exclude_none=True))
    return HttpResultResponse(total=num, data=res)


@router.post("/submit/fixTable")
def finance_fix(data: acc_schemas.FixTable, request: Request,
                token=Depends(login_required), db: Session = Depends(get_db)):
    """新增修复报表"""
    header_list = request.get("headers")
    res = guild.create_fix_table(db, data, header_list)
    return HttpResultResponse(data=res.id)


@router.post("/data/recovery")
def recovery_fix(data: acc_schemas.Recovery_fix, token=Depends(login_required), db: Session = Depends(get_db)):
    """异常数据修复"""
    recovery_list, total = guild.recovery_fix_data(db, Account_log, data.dict(exclude_none=True))
    return HttpResultResponse(total=total, data=recovery_list)


@router.post("/submit/recovery")
def finance_fix(data: acc_schemas.RecoveryTable, token=Depends(login_required), db: Session = Depends(get_db)):
    """新增异常数据修复"""
    res = guild.create_recovery_table(db, data)
    return HttpResultResponse(data=res.id)


@router.post("/recovery/fix")
def recovery_fix(data: acc_schemas.RecoveryupdateTable, token=Depends(login_required), db: Session = Depends(get_db)):
    """异常数据修复"""
    res = guild.update_recovery_table(db, data.dict(exclude_none=True))
    return HttpResultResponse(data=res.get("id"))
