import json
from typing import Optional
from fastapi import Depends, APIRouter, Request, Query
from sqlalchemy.orm import Session
from app import get_db
from app.api.account import schemas, crud
from app.api.account.crud import AccountStatistics, SpecificAccountQuery
from app.api.statement import crud as statement_crud
from libs import functions
from libs.functions import get_date_list
from libs.result_format import HttpResultResponse, HttpMessage
from libs.token_verify import login_required

router = APIRouter()


@router.post("/create")
def create_account(data: schemas.AccountCreate, token=Depends(login_required)):
    """添加账户"""
    db_info = crud.get_account(data.name)
    if db_info:
        return HttpResultResponse(code=400, msg=HttpMessage.ACCOUNT_EXIST)
    res = crud.create_account(data)
    if res:
        return HttpResultResponse(code=500, msg=res)
    return HttpResultResponse()


@router.get("/list")
def user_list(page: int, size: int, name: Optional[str] = None, token=Depends(login_required)):
    """账户列表"""
    result, num = crud.HDUd().get_account_list(name, page, size)
    return HttpResultResponse(total=num, data=result)


@router.get("/gift/type")
def gift_type_list(unique_tag: Optional[str] = "", token=Depends(login_required)):
    """礼物类型配置返回"""
    result = crud.get_gift_type(unique_tag)
    return HttpResultResponse(data=result)


@router.post("/update")
def read_account(data: schemas.AccountUpdate, token=Depends(login_required)):
    """账户修改"""
    if not data.name:
        return HttpResultResponse(code=500, msg="账户名不能为空")
    crud.update_account_info(data)
    return HttpResultResponse()


@router.get("/finance/info")
def finance_information(unique_tag: str,
                        page: int,
                        size: int,
                        id: Optional[int] = None,
                        start_time: Optional[str] = "",
                        end_time: Optional[str] = "",
                        token=Depends(login_required)):
    # unique_tag:Optional[str] = Query(None, min_length=3, max_length=50, regex="^xiao\d+$") 参数验证
    """账户财务详情列表"""
    res, total = crud.get_finance_info(unique_tag, id, page, size, start_time, end_time, 1)
    return HttpResultResponse(total=total, data=res)


@router.post("/finance/info/excel")
def finance_info_excel(data: schemas.FinanceInfo, request: Request,
                       token=Depends(login_required), db: Session = Depends(get_db)):
    """账户财务详情导出"""
    headers = request.get("headers")
    statement_list = crud.get_finance_info(data.unique_tag, data.id, data.page, 99999, data.start_time, data.end_time)
    df = ['账户余额', '入账', '出账', '时间']
    return statement_crud.data_to_file(db, statement_list, "财务信息", headers, df)


@router.get("/finance/details")
def finance_details(page: int,
                    size: int,
                    uuid: str,
                    user_id: Optional[str] = "",
                    start_time: Optional[str] = "",
                    end_time: Optional[str] = "",
                    type: Optional[int] = None,
                    gift_type: Optional[str] = "",
                    unique_tag: Optional[str] = "",
                    token=Depends(login_required)):
    """账户财务明细列表"""
    if not start_time and not end_time:
        return HttpResultResponse(code=500, msg="请输入你要查询的时间段")
    res, total, count = AccountStatistics(page, size, uuid, user_id, start_time, end_time, type, gift_type,
                                          unique_tag).get_finance_details(is_list=1)
    return HttpResultResponse(total=total, data=res, count=count)


@router.post("/finance/details/excel")
def finance_info_excel(data: schemas.FinanceDetails, request: Request,
                       token=Depends(login_required), db: Session = Depends(get_db)):
    """账户财务明细导出"""
    headers = request.get("headers")
    statement_list = AccountStatistics(data.page, 99999, data.uuid, data.user_id, data.start_time, data.end_time, data.type,
                                       data.gift_type, data.unique_tag).get_finance_details()
    data = crud.delete_specify_field(statement_list)
    field_head = ['订单号', '出入账方式', '礼物类型', '金额', '时间']
    return statement_crud.data_to_file(db, data, "财务明细", headers, field_head)


@router.get("/finance/fourth/details")
def finance_fourth_info(page: int,
                        size: int,
                        uuid: Optional[str] = "",
                        guild_id: Optional[str] = "",  # 针对公会账户，没有uuid，传公会id过来，再查uuid
                        time: Optional[str] = "",
                        type: Optional[int] = None,
                        reference_type: Optional[str] = "",
                        unique_tag: Optional[str] = "",
                        token=Depends(login_required)):
    """账户财务明细  第四层"""
    if not all([time, unique_tag]):
        return HttpResultResponse(code=500, msg="缺少必传参数")
    res, total, count = SpecificAccountQuery(page, size, uuid, time, type, reference_type, unique_tag,
                                             guild_id).business_logic()
    return HttpResultResponse(total=total, data=res, count=count)


@router.get("/type")
def finance_fix(page: int,
                size: int,
                key_name: str = "",
                key_value: str = "",
                type: int = None,
                token=Depends(login_required), db: Session = Depends(get_db)):
    """出入账目配置列表"""
    res, num = crud.get_account_type(key_name=key_name, key_value=key_value, type=type, page=page, size=size)
    return HttpResultResponse(total=num, data=res)


@router.post("/type/update")
def finance_fix(data: schemas.CreateType, token=Depends(login_required), db: Session = Depends(get_db)):
    """出入账目类型修改"""
    if not all([data.keyName, data.keyValue]) or data.type is None:
        return HttpResultResponse(code=500, msg="缺少必传参数")
    res = crud.update_account_type(data)
    if res:
        return HttpResultResponse()
    return HttpResultResponse(code=422, msg='修改失败')


@router.post("/create/type")
def create_user(data: schemas.CreateType, token=Depends(login_required), db: Session = Depends(get_db)):
    """添加账目类型"""
    if not all([data.keyName, data.keyValue]) or data.type is None:
        return HttpResultResponse(code=500, msg="缺少必传参数")
    res_type = crud.get_account_type_value(data.keyValue, data.type)
    if res_type:
        return HttpResultResponse(code=400, msg=HttpMessage.TYPE_EXIST)
    res = crud.create_type(data)
    if not res:
        return HttpResultResponse(code=500, msg=res)
    return HttpResultResponse()


@router.get("/userInfo")
def query_users_info(uuid: str, token=Depends(login_required)):
    """查询用户信息"""
    res = functions.query_user_info(uuid)
    if res.status_code != 200:
        return HttpResultResponse(code=500, msg=str(res))
    success = json.loads(res.text)
    return HttpResultResponse(data=success.get('data'))


@router.get("/guildInfo")
def query_guilds_info(uuid: str, token=Depends(login_required)):
    """查询用户信息"""
    res = functions.query_guild_info(uuid)
    if res.status_code != 200:
        return HttpResultResponse(code=500, msg=str(res))
    success = json.loads(res.text)
    return HttpResultResponse(data=success.get('data'))
