from decimal import Decimal

from fastapi import HTTPException
from typing import Optional
from pydantic import BaseModel, validator


class StatementBase(BaseModel):
    user_id: int
    zb_number: int
    order_number: int


class StatementCreate(StatementBase):
    nickname: str
    recharge_amount: float
    recharge_channel: str
    flowing_water_number: int


class StatementList(BaseModel):
    page: int = 1
    size: int = 9999999
    start_time: Optional[str] = ""
    end_time: Optional[str] = ""
    order_number: Optional[str] = ""
    uuid: Optional[str] = ""
    sid: Optional[str] = ""
    types: Optional[str] = ""
    month_type: Optional[int] = None
    menu_id: Optional[list] = ""


class Statement(StatementBase):
    id: int

    class Config:
        orm_mode = True


class UserWithdrawalList(BaseModel):
    page: Optional[int] = None
    size: Optional[int] = None
    start_time: Optional[str] = ""
    end_time: Optional[str] = ""
    status: Optional[int] = None
    name: Optional[str] = ""


class GuildWithdrawalList(BaseModel):
    page: int
    size: int
    start_time: str = None
    end_time: str = None
    status: int = None
    guild_id: int = None


class PaymentWithdrawalList(BaseModel):
    page: int = None
    size: int = None
    start_time: str = None
    end_time: str = None
    status: int = None
    guild_id: int = None

    @validator('start_time')
    def validate_start_time(cls, val):
        if val != '':
            return val

    @validator('end_time')
    def validate_end_time(cls, val):
        if val != '':
            return val

    @validator('status')
    def validate_status(cls, val):
        if val != '':
            return val

    @validator('guild_id')
    def validate_guild_id(cls, val):
        if val != '':
            return val


class PaymentAdd(BaseModel):
    uuid: str
    money: float
    cont: str
    url: str

    @validator('money')
    def money_must_contain_space(cls, c):
        if c:
            c = round(float(c), 3)
        return c

    def __str__(self):
        return "uuid：%s, money：%s ,cont:%s ,url:%s" % (str(self.uuid), self.money, self.cont, self.url)


class PaymentAccountlList(PaymentWithdrawalList):
    uuid: int = None
    transfer_name: Optional[str] = ""
    dst_name: Optional[str] = ""


class GuildUpdate(BaseModel):
    id: int
    status: int


class UserNumber(BaseModel):
    id: Optional[int] = None
    uuid: Optional[str] = ""
    user_number: Optional[int] = None
    type: Optional[int] = None


class CreateBill(BaseModel):
    dst_id: Optional[str] = ""
    dst_name: Optional[str] = ""
    amount: Optional[float] = 0.0
    transfer_id: Optional[str] = ""
    remark: Optional[str] = ""
    transfer_type: Optional[str] = ""
    type: Optional[int] = None
    amount_type: Optional[str] = ""

    # @validator('transfer_id')
    # def transfer_id_must_contain_space(cls, v):
    #     if not v:
    #         raise ValueError('缺少必传参数')
    #     return v
    #
    # @validator('remark')
    # def remark_must_contain_space(cls, v):
    #     if not v:
    #         raise ValueError('缺少必传参数remark')
    #     return v


class MenuAdd(BaseModel):
    menu_name: str
    menu_label: str
    menu_type: int
    remark: Optional[str] = None


class MenuEdit(BaseModel):
    id: str
    menu_name: str
    menu_label: str
    menu_type: int
    remark: Optional[str] = None


class GuildAddLog(BaseModel):
    id: int
    guild_id: Optional[int] = None
    balance: Optional[str] = None
    pearl: Optional[str] = None
    status: int
    create_time: Optional[int] = None
    update_time: Optional[str] = None
    guild_name: Optional[str] = None
    guild_per: Optional[str] = None
    actual_money: Optional[str] = None
    pledge_deposit_money: Optional[str] = None


class GuildModifyLog(BaseModel):
    guild: Optional[list] = None
