import math
import threading
from sqlalchemy import and_, func
from sqlalchemy.orm import Session

from libs.functions import get_now_timestamp, get_now_datetime
from libs.orm import QueryAllData
from libs.token_verify import get_current_user
from models.recharge import Settlement, Fitransferlog, FinanceFixLog
from models.users import V2User
from models.guild import Guild
from models.account import Account

locka = threading.Lock()


class GuildSet(object):
    def __init__(self):
        self.settlement_list = []

    def dispose_guild_data(self, db, num):
        guild_list = []
        locka.acquire()
        once_res = db.query(Settlement).filter().offset(num * 10).limit(10).all()
        locka.release()
        for i in once_res:
            info_dict = i.to_dict()
            guild_list.append(info_dict)
        self.settlement_list += guild_list

    def query_guild_data(self, db):
        msg_count = db.query(func.count(Settlement.id)).scalar()
        num = math.ceil(msg_count / 10)
        # 创建线程
        ths = []
        for x in range(num):
            ths.append(threading.Thread(target=self.dispose_guild_data, args=[db, x]))
        for y in range(num):
            ths[y].start()
        for z in range(num):
            ths[z].join()
        return self.settlement_list

    def get_guild_data(self, db: Session, param):
        """公会结算"""
        is_guild_filters = []
        if param.guild_id:
            is_guild_filters.append(Settlement.guild_id == param.guild_id)
        if param.status or param.status == 0:
            is_guild_filters.append(Settlement.status == param.status)
        if param.start_time:
            is_guild_filters.append(Settlement.create_time >= param.start_time)
        if param.end_time:
            is_guild_filters.append(Settlement.create_time <= param.end_time)
        # 判断有无条件
        try:
            if len(is_guild_filters) > 0:
                get_user_orm_sql = db.query(Settlement).filter(and_(*is_guild_filters))
                condition_data = db.execute(get_user_orm_sql).fetchall()
                guild_info = [i[0].to_dict() for i in condition_data]
            else:
                guild_info = self.query_guild_data(db)
        except Exception as e:
            return [], 0
        # 判断是列表还是导出接口
        guild_info.reverse()
        if guild_info:
            return guild_info[(int(param.page) - 1) * param.size:param.size * param.page], len(guild_info)
        return [], 0

    def update_guild_info(self, db: Session, data):
        """公会结算修改"""
        if (data.status or data.status == 0) and data.status < 2:
            db.query(Settlement).filter(Settlement.id == data.id).update({Settlement.status: data.status + 1})
            db.commit()


def paymentset_guild_data(db: Session, dbname, params):
    paymentset_filters = []
    if params.get("id"):
        paymentset_filters.append(dbname.guild_id == params.get("guild_id"))
    if params.get("cont"):
        paymentset_filters.append(dbname.cont == params.get("cont"))
    if params.get("money"):
        paymentset_filters.append(dbname.money == params.get("money"))
    if params.get("url"):
        paymentset_filters.append(dbname.url == params.get("url"))
    if params.get("uuid"):
        paymentset_filters.append(dbname.uuid == params.get("uuid"))
    if params.get("start_time"):
        paymentset_filters.append(dbname.create_time >= params.get("start_time"))
    if params.get("end_time"):
        paymentset_filters.append(dbname.create_time <= params.get("end_time"))
    querydata, count = QueryAllData(db, dbname, params, paymentset_filters).query_data()
    data = [QueryAllData.serialization(item) for item in querydata]
    return data, count


def outon_account_data(db: Session, dbname, params):
    querydata, count = QueryAllData(db, dbname, params, None).query_data()
    data = [QueryAllData.serialization(item,
                                       remove={'operator', 'unique_tag', 'create_time', 'beneficiary', 'description',
                                               'create_time', 'config_key', 'income', 'output', 'operator_id'}) for item
            in querydata]
    return data, count


def accout_list_data(db: Session, dbname, params):
    accout_filters = []
    if params.get("id"):
        accout_filters.append(dbname.guild_id == params.get("guild_id"))
    if params.get("transfer_id"):
        accout_filters.append(dbname.transfer_id == params.get("transfer_id"))
    if params.get("dst_id"):
        accout_filters.append(dbname.dst_id == params.get("dst_id"))
    if params.get("cont"):
        accout_filters.append(dbname.cont == params.get("cont"))
    if params.get("money"):
        accout_filters.append(dbname.money == params.get("money"))
    if params.get("url"):
        accout_filters.append(dbname.url == params.get("url"))
    if params.get("uuid"):
        accout_filters.append(dbname.uuid == params.get("uuid"))
    if params.get("start_time"):
        accout_filters.append(dbname.create_time >= params.get("start_time") + " 00:00:00")
    if params.get("end_time"):
        accout_filters.append(dbname.create_time <= params.get("end_time") + " 23:59:59")
    querydata, count = QueryAllData(db, dbname, params, accout_filters).query_data()
    data = [QueryAllData.serialization(item) for item in querydata]
    return data, count


def query_uuid_or_user_number(db: Session, param):
    """查询uuid和user_number"""
    if param.uuid:
        user = db.query(V2User).filter(V2User.uuid == param.uuid).first()
        if not user:
            guild = db.query(Guild).filter(Guild.uuid == param.uuid).first()
            if not guild:
                return {}
            return {"nick_name": "(公会)" + guild.guild_name, "user_number": guild.id, "is_guild": 1}
        return {"nick_name": user.nick_name, "user_number": user.user_id, "is_guild": 0}
    if param.user_number:
        user = db.query(V2User).filter(V2User.user_number == param.user_number).first()
        return {"nick_name": user.nick_name, "user_id": user.user_id, "uuid": user.uuid}


def account_money(db: Session, param):
    """查询账户余额"""
    account = db.query(Account).filter(Account.id == param.id).first()
    if not account:
        return 0
    return account.consumable if account.consumable else 0


def query_token(db, h_list):
    """获取请求头token"""
    token = ''
    for k, v in dict(h_list).items():
        key = k.decode('utf-8')
        if key == 'authorization':
            tokens = v.decode('utf-8')
            _, token = tokens.split(' ')
            break
    # 通过token获取用户信息
    user_info = get_current_user(db=db, token=token)
    return user_info


def transfer_money(db: Session, param, h_list):
    """转账"""
    user = query_token(db, h_list)
    try:
        present = Fitransferlog(dst_id=param.dst_id, amount=param.amount, type=param.type,
                                transfer_id=param.transfer_id, nick_name=param.nick_name, remark=param.remark,
                                operator=user.get("username"), transfer_type=param.transfer_type,
                                create_time=get_now_timestamp())
        db.add(present)
        db.commit()
        db.refresh(present)
    except Exception as e:
        return {}
    return present


def create_fix_table(db: Session, param, h_list):
    """增加修复报表"""
    user = query_token(db, h_list)
    if param.type == 0:
        param.amount_type = 'consumable'
    try:
        present = FinanceFixLog(type=param.type, account_id=param.account_id, money=param.money,
                                remark=param.remark, operator=user.get("username"), amount_type=param.amount_type,
                                create_time=get_now_datetime())
        db.add(present)
        db.commit()
        db.refresh(present)
    except Exception as e:
        return {}
    return present
