from datetime import datetime
from dateutil.relativedelta import relativedelta
from fastapi import Depends, APIRouter
from sqlalchemy.orm import Session
from app import get_db
from app.api.export import schemas, crud
from libs.result_format import HttpResultResponse, HttpMessage

router = APIRouter()


@router.post("/list")
def export_list(data: schemas.ExportList, db: Session = Depends(get_db)):
    """导出记录列表"""
    result, total = crud.get_export_list(db, data.dict(exclude_none=True))
    return HttpResultResponse(total=total, data=result)


@router.post("/update")
def update_export_list(data: schemas.ExportUpdateData, db: Session = Depends(get_db)):
    """导出记录修改"""
    result = crud.update_export(db, data)
    if result:
        return HttpResultResponse()
    return HttpResultResponse(code=422, msg=HttpMessage.LOSE)


@router.delete("/delete/{id}")
def delete_export_one(id:int, db: Session = Depends(get_db)):
    """导出记录删除"""
    is_delete = crud.delete_export(db, id)
    if is_delete:
        return HttpResultResponse()
    return HttpResultResponse(code=422, msg=HttpMessage.DELETE_LOSE)


@router.get("/source")
def export_source_query(db: Session = Depends(get_db)):
    """导出来源查询"""
    result = crud.get_source_data(db)
    return HttpResultResponse(data=result)


@router.post("/month/total")
def month_query_total_export(param: schemas.MonthStatistics):
    """月度表计算"""
    if not param.date:
        month_date = datetime.now().date() - relativedelta(months=1)
        param.date = month_date.strftime("%Y%m")
    result, num = crud.CalculationMonthlyBill().month_statistics_task('assets_log_' + param.date, param.type,
                                                                      param.page, param.size)
    return HttpResultResponse(total=num, data=result)


@router.post("/month/details")
def month_query_total_export(param: schemas.MonthDetails):
    """月度计算,详情"""
    if not param.date:
        return HttpResultResponse(msg='查询月份不能为空')
    result, num = crud.CalculationMonthlyDetails().statement_income_expenditure(param)
    return HttpResultResponse(total=num, data=result)


@router.post("/month")
def month_to_export(param: schemas.MonthStatistics):
    """月度表导出"""
    if not param.date:
        return HttpResultResponse(msg='查询月份不能为空')
    result = crud.MonthDataDerive().derive_data('assets_log_' + param.date, param.type)
    return result
